/* dinaao.c, written by John Gilbert */
/* Version 1.01 */
/* Copyright 2008 */

/*

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/


#include <stdio.h> /*getchar*/
#include <stdlib.h> /*atexit*/
#include <unistd.h> /*sleep*/
#include <string.h> /*memcpy*/
#include <termios.h> /*tcgetattr*/

#include "dinaao.h"
#include "input.h"

#ifdef DEBUG
int trace_cpu = 1;
int use_ansi_color = 0;
int throttle = 0;
#else
int trace_cpu = 0;
int use_ansi_color = 0;
int throttle = 0;
#endif

int dopause = 0;
int dostep = 0;

void do_init(void);
void cpu_init(void);
void do_cpu(void);
void do_output(void);
void do_exit(void);
void do_banner(void);
void read_raw(unsigned char *, char *, int);

/* 6502 stuff finally */
unsigned char *mem_map;
#define MEMSIZE 0x10000
typedef struct
{
	unsigned char P; /* processor status (flags) = NVXBDICZ */
	unsigned char A; /* Accumulator */
	unsigned char X; /* X register */
	unsigned char Y; /* Y register */
	unsigned char SP; /* Stack point, points to 0x100 - 0x1FF */
	unsigned short PC; /* Program Counter */
} m6502_regs;
m6502_regs m6502;

/*
 * Processor Status (flags)
 * N = Negitive
 * V = Overflow
 * X = always 1, unused
 * B = Break, set by BRK, unset by IRQ, what resets this? 1=BRK, 0=IRQB
 * D = Decimal mode, not always cleared on reset on 6502, cleared on BRK.
 * I = IRQ Disable, set by BRK
 * Z = Zero
 * C = Carry
 */

#define N_FLAG 0x80
#define V_FLAG 0x40
#define X_FLAG 0x20
#define B_FLAG 0x10
#define D_FLAG 0x08
#define I_FLAG 0x04
#define Z_FLAG 0x02
#define C_FLAG 0x01


/* Set by program to affect cpu */
#define DO_NMI	 0x01 /* non-maskable interrupt */
#define DO_RESET 0x02 /* restart cpu */ 
#define DO_IRQ	 0x04 /* interrupt, ignored if I_FLAG is set */
#define DO_STOV	 0x08 /* SOB, set overflow bit, toggle */

/********************
 * Addressing Modes *
 ********************/
/* *(*arg+y) | (zp),y */
/* Take the address at arg, and arg+1, add Y, that's where you go. */
/* arg can't be 0xFF or ? */
#define INDIRECTY mem_map[((0xFF & mem_map[mem_map[m6502.PC + 1] + 1]) << 8) + \
	mem_map[mem_map[m6502.PC + 1]] + m6502.Y]

/* **(arg + x) | (zp,x) */
/* Zero Page Indexed Indirect: (zp,x), Pre indexed

The value in X is added to the specified first page address for a sum address. The address at the sum address and the sum address plus one is loaded and the value at that address is loaded into the Accumulator.

Example

The value $02 in X is added to $15 for a sum of $17. The address $D010 at addresses $0017 and $0018 is loaded and the value $33 at $D010 is loaded into the Accumulator.

LDA (15,X)
	X = 02
zero page $15 + 2 = $17
look at $17 and $18 for address D010
value at D010 -> A
*/

#define INDIRECTX mem_map[ \
	(mem_map[mem_map[m6502.PC + 1] + m6502.X + 1] << 8) + \
	mem_map[mem_map[m6502.PC + 1] + m6502.X]]

/* *(arg + x) */
#define ABSOLUTEX mem_map[(mem_map[m6502.PC + 2] << 8) + \
	mem_map[m6502.PC + 1] + m6502.X]

/* *(arg + y) */
#define ABSOLUTEY mem_map[(mem_map[m6502.PC + 2] << 8) + \
	mem_map[m6502.PC + 1] + m6502.Y]

/* *arg */
#define ABSOLUTE mem_map[(mem_map[m6502.PC + 2] << 8) + mem_map[m6502.PC + 1]]

/* *(arg + x & 0xFF) */
#define ZEROPAGEX mem_map[(mem_map[m6502.PC + 1] + m6502.X) & 0xFF]

/* *(arg + y & 0xFF) */
#define ZEROPAGEY mem_map[(mem_map[m6502.PC + 1] + m6502.Y) & 0xFF]

/* *arg */
#define ZEROPAGE mem_map[mem_map[m6502.PC + 1]]

/* arg */
#define IMMEDIATE mem_map[m6502.PC + 1]

/* and new for 65C02 */
/* Zeropage Indirect */
#define INDIRECTZ mem_map[((0xFF & mem_map[mem_map[m6502.PC + 1] + 1]) << 8) + \
	mem_map[mem_map[m6502.PC + 1]]]

/* Absolute Indexed Indirect, used only for JMP */

/* relative and implied are coded directly */

/***************************
 * End of Addressing Modes *
 ***************************/

unsigned char pending_irq = 0; /* pending interrupts */
unsigned int clockcnt = 0;

int
main(int argc, char **argv)
{
	do_init();
	for(;;) {
		do_cpu();
	}
}

void
do_init(void)
{
	int x;
	cpu_init();
	kb_init();
	do_banner();
	atexit(do_exit);
	if ((mem_map=(unsigned char *)malloc(MEMSIZE)) == NULL) exit(-1);
	for (x = 0; x < MEMSIZE; x++) mem_map[x] = 0x00;
/*	printf("Loading wozmon.bin... -> 0xFF00 - 0xFFFF\n");
	read_raw(&mem_map[0xFF00], "wozmon.bin", 0x0100); */

/*	printf("Loading replica1.bin... -> 0xE000 - 0xFFFF\n");
	read_raw(&mem_map[0xE000], "replica1.bin", 0x2000); */

	printf("Loading replica13a.bin... -> 0xE000 - 0xFFFF\n");
	read_raw(&mem_map[0xE000], "replica13a.bin", 0x2000);

/*	printf("Loading cffa1.bin... -> 0x9000 - 0xAFFF\n");
	read_raw(&mem_map[0x9000], "cffa1.bin", 0x2000); */
	
/*	printf("Loading cassette.bin... -> 0xC100 - 0xC1FF\n");
	read_raw(&mem_map[0xC100], "cassette.bin", 0x0100); */

	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("========================================\n");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */

	/* clear IO area */
	mem_map[0xD010] = 0x80; /* high bit constantly on */
	mem_map[0xD011] = 0x00;
	mem_map[0xD012] = 0x00;
	mem_map[0xD013] = 0x00;

	/* Startup input thread */
	if (pthread_create(&keyinthread, NULL, InputThread, NULL) == -1) {
		printf("Error: unable to create input thread.\n");
		exit(1);
	}

	/* set up condition variables and mutex lock */
	pthread_cond_init(&full, NULL);	/* no characters stored yet */
	pthread_cond_init(&empty, NULL); /* all character slots are empty */
	pthread_mutex_init(&que_lock,NULL);

}

void
do_banner(void)
{
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("Welcome to Dinaao.\n");
	printf("Written by John Gilbert.\n");
	printf("Hit F1 for Help, F9 to Exit, F12 to Reset.\n");
	printf("========================================\n");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
}

void
do_exit(void)
{
	kb_exit();
	if (mem_map) {
		free(mem_map);
		mem_map = NULL;
	}

}

void
cpu_init(void)
{
	m6502.PC = 0xFF00; /* does this start this way? */
	/* if not so, how does the stack look at start? */
	m6502.SP = 0xFF; /* set to top of stack, +0x100 implied, increments downward */
	m6502.X = m6502.Y = m6502.A = 0;
	/* set break and unused flag, BRK will unflag B */
	m6502.P = X_FLAG | B_FLAG | Z_FLAG;
}

void
do_cpu(void)
{
	unsigned short tmpval;
#ifdef DEBUG
	static unsigned short oldstack = 0x0000;
#endif /* DEBUG */

	if (dopause) return;

#ifdef DEBUG
	if ((trace_cpu >= 2) && (oldstack != ((mem_map[0x0101] << 8) + mem_map[0x100]))) {
		oldstack = (mem_map[0x0101] << 8) + mem_map[0x100];
		printf("\nstackstart is now = %04X\n", oldstack);
	}
#endif /* DEBUG */


	/* set or clear keyboard strobe */
	mem_map[0xD011] = IsKey();

	/* handle reset, and then print trace */
	if (pending_irq & DO_RESET) {
		cpu_init(); /* is this right? */
	/* clear the reset */
		pending_irq &= ~DO_RESET;
	/*	m6502.SP = 0xFF; * back to top of stack */
	/*	m6502.P = (X_FLAG | I_FLAG); * reset the flags */
	/*	m6502.PC = (mem_map[0xFFFD] << 8) + mem_map[0xFFFC]; */
		if (trace_cpu) fprintf(stderr, "<RESET>");
	}

	/* TODO: handle NMI or IRQ */


	/* next print out trace */
	if (trace_cpu >= 1) {
		printf("\nPC:%04X @:%02X SP:1%02X A:%02X X:%02X Y:%02X "
			"%c%c%c%c%c%c%c%c"
#ifdef NOTDEF
			" clock=%05u KBR=%02X KBCR=%02x XAM=%02X%02X "
/*			" 1F9:%02X 1F8:%02X 104:%02X 105:%02X " */
#endif
			, m6502.PC, mem_map[m6502.PC], m6502.SP,

#ifdef NOTDEF /* stack pointer value */
			mem_map[0x100 + m6502.SP],
#endif
			m6502.A, m6502.X, m6502.Y,
			(m6502.P & N_FLAG) ? 'N' : '.',
			(m6502.P & V_FLAG) ? 'V' : '.',
			(m6502.P & X_FLAG) ? 'R' : '.',
			(m6502.P & B_FLAG) ? 'B' : '.',
			(m6502.P & D_FLAG) ? 'D' : '.',
			(m6502.P & I_FLAG) ? 'I' : '.',
			(m6502.P & Z_FLAG) ? 'Z' : '.',
			(m6502.P & C_FLAG) ? 'C' : '.'
#ifdef NOTDEF
			, clockcnt, mem_map[0xD010], mem_map[0xD011],
			mem_map[0x0025], mem_map[0x0024]
	/*		mem_map[0x01F9], mem_map[0x01F8],
			mem_map[0x0104], mem_map[0x0105] */
#endif
			);
	}

	if (throttle) sleep(1); /* one second for now */


	/* third, do an operation */
	switch(mem_map[m6502.PC]) {

/***********************
 * Branch Instructions *
 ***********************/

		case 0x10: /* BPL (branch if plus (not negative)) */
	/* note: takes 2 clocks if not branched, 3 if branched to same page, 4 if next page */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tBPL #%02X %s ", mem_map[m6502.PC + 1], (m6502.P & N_FLAG) ? "No" : "Yes");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (!(m6502.P & N_FLAG)) {
				clockcnt++;
				if ((m6502.PC & 0xFF00) != ((m6502.PC + mem_map[m6502.PC + 1]) & 0xFF00))
					clockcnt++;
				/* note: use signed value, might jump backwards */
				m6502.PC += 2 + (char) mem_map[m6502.PC + 1];
			} else
				m6502.PC += 2;
			clockcnt += 2;
			break;

		case 0x30: /* BMI (branch on minus (negative)) */
	/* note: takes 2 clocks if not branched, 3 if branched to same page, 4 if next page */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tBMI #%02X %s ", mem_map[m6502.PC + 1], (m6502.P & N_FLAG) ? "Yes" : "No");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (m6502.P & N_FLAG) {
				clockcnt++;
				if ((m6502.PC & 0xFF00) != ((m6502.PC + mem_map[m6502.PC + 1]) & 0xFF00))
					clockcnt++;
				/* note: use signed value, might jump backwards */
				m6502.PC += 2 + (char) mem_map[m6502.PC + 1];
			} else
				m6502.PC += 2;
			clockcnt += 2;
			break;

		case 0x50: /* BVC (branch on oVerflow clear) */
	/* note: takes 2 clocks if not branched, 3 if branched to same page, 4 if next page */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tBVC #%02X %s ", mem_map[m6502.PC + 1], (m6502.P & V_FLAG) ? "No" : "Yes");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (!(m6502.P & V_FLAG)) {
				clockcnt++;
				if ((m6502.PC & 0xFF00) != ((m6502.PC + mem_map[m6502.PC + 1]) & 0xFF00))
					clockcnt++;
				/* note: use signed value, might jump backwards */
				m6502.PC += 2 + (char) mem_map[m6502.PC + 1];
			} else
				m6502.PC += 2;
			clockcnt += 2;
			break;

		case 0x70: /* BVS (branch on oVerflow set) */
	/* note: takes 2 clocks if not branched, 3 if branched to same page, 4 if next page */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tBVS #%02X %s ", mem_map[m6502.PC + 1], (m6502.P & V_FLAG) ? "Yes" : "No");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (m6502.P & V_FLAG) {
				clockcnt++;
				if ((m6502.PC & 0xFF00) != ((m6502.PC + mem_map[m6502.PC + 1]) & 0xFF00))
					clockcnt++;
				/* note: use signed value, might jump backwards */
				m6502.PC += 2 + (char) mem_map[m6502.PC + 1];
			} else
				m6502.PC += 2;
			clockcnt += 2;
			break;

		case 0x90: /* BCC (branch if carry clear) */
	/* note: takes 2 clocks if not branched, 3 if branched to same page, 4 if next page */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tBCC #%02X %s ", mem_map[m6502.PC + 1], (m6502.P & C_FLAG) ? "No" : "Yes");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (!(m6502.P & C_FLAG)) {
				clockcnt++;
				if ((m6502.PC & 0xFF00) != ((m6502.PC + mem_map[m6502.PC + 1]) & 0xFF00))
					clockcnt++;
				/* note: use signed value, might jump backwards */
				m6502.PC += 2 + (char) mem_map[m6502.PC + 1];
			} else
				m6502.PC += 2;
			clockcnt += 2;
			break;

		case 0xB0: /* BCS (branch if carry set) */
	/* note: takes 2 clocks if not branched, 3 if branched to same page, 4 if next page */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tBCS #%02X %s ", mem_map[m6502.PC + 1], (m6502.P & C_FLAG) ? "Yes" : "No");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (m6502.P & C_FLAG) {
				clockcnt++;
				if ((m6502.PC & 0xFF00) != ((m6502.PC + mem_map[m6502.PC + 1]) & 0xFF00))
					clockcnt++;
				/* note: use signed value, might jump backwards */
				m6502.PC += 2 + (char) mem_map[m6502.PC + 1];
			} else
				m6502.PC += 2;
			clockcnt += 2;
			break;

		case 0xD0: /* BNE (branch if not equal (zero not set)) */
	/* note: takes 2 clocks if not branched, 3 if branched to same page, 4 if next page */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tBNE #%02X %s ", mem_map[m6502.PC + 1], (m6502.P & Z_FLAG) ? "No" : "Yes");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (!(m6502.P & Z_FLAG)) {
				clockcnt++;
				if ((m6502.PC & 0xFF00) != ((m6502.PC + mem_map[m6502.PC + 1]) & 0xFF00))
					clockcnt++;
				/* note: use signed value, might jump backwards */
				m6502.PC += 2 + (char) mem_map[m6502.PC + 1];
			} else
				m6502.PC += 2;
			clockcnt += 2;
			break;

		case 0xF0: /* BEQ (branch if equal (zero set)) */
	/* note: takes 2 clocks if not branched, 3 if branched to same page, 4 if next page */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tBEQ #%02X %s ", mem_map[m6502.PC + 1], (m6502.P & Z_FLAG) ? "Yes" : "No");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (m6502.P & Z_FLAG) {
				clockcnt++;
				if ((m6502.PC & 0xFF00) != ((m6502.PC + mem_map[m6502.PC + 1]) & 0xFF00))
					clockcnt++;
				/* note: use signed value, might jump backwards */
				m6502.PC += 2 + (char) mem_map[m6502.PC + 1];
			} else
				m6502.PC += 2;
			clockcnt += 2;
			break;

/******************************
 * End of Branch Instructions *
 ******************************/

/*********************
 * Flag Instructions *
 *********************/

		case 0x18: /* CLC (CLear Carry) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tCLC ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~C_FLAG;
			m6502.PC += 1;
			clockcnt += 2;
			break;

		case 0x38: /* SEC (SEt Carry) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSEC ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P |= C_FLAG;
			m6502.PC += 1;
			clockcnt += 2;
			break;

		case 0x58: /* CLI (Clear Interrupt) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tCLI ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~I_FLAG;
			m6502.PC += 1;
			clockcnt += 2;
			break;

		case 0x78: /* SEI (Set Interrupt) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSEI ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P |= I_FLAG;
			m6502.PC += 1;
			clockcnt += 2;
			break;

		case 0xB8: /* CLV (Clear Overflow) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tCLV ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~V_FLAG;
			m6502.PC += 1;
			clockcnt += 2;
			break;

		case 0xD8: /* CLD (Clear Decimal) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tCLD ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~D_FLAG;
			m6502.PC += 1;
			clockcnt += 2;
			break;

		case 0xF8: /* SED (Set Decimal) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSED ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P |= D_FLAG;
			m6502.PC += 1;
			clockcnt += 2;
			break;

/****************************
 * End of Flag Instructions *
 ****************************/

/*************************
 * Register Instructions *
 *************************/

		case 0xAA: /* TAX (transfer A to X) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tTAX ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.X = m6502.A;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.X) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.X; /* is it negitive? */
			m6502.PC++;
			clockcnt += 2;
			break;

		case 0x8A: /* TXA (transfer X to A) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tTXA ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A = m6502.X;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.X) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.X; /* is it negitive? */
			m6502.PC++;
			clockcnt += 2;
			break;

		case 0xCA: /* DEX */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tDEX ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.X--;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.X) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.X; /* is it negitive? */
			m6502.PC++;
			clockcnt += 2;
			break;

		case 0xE8: /* INX */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tINX ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.X++;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.X) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.X; /* is it negitive? */
			m6502.PC++;
			clockcnt += 2;
			break;

		case 0xA8: /* TAY (transfer A to Y) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tTAY ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.Y = m6502.A;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.Y) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.Y; /* is it negitive? */
			m6502.PC++;
			clockcnt += 2;
			break;

		case 0x98: /* TYA (transfer Y to A) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tTYA ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A = m6502.Y;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC++;
			clockcnt += 2;
			break;

		case 0x88: /* DEY */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tDEY ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.Y--;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.Y) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.Y; /* is it negitive? */
			m6502.PC++;
			clockcnt += 2;
			break;

		case 0xC8: /* INY */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tINY ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.Y++;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.Y) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.Y; /* is it negitive? */
			m6502.PC++;
			clockcnt += 2;
			break;

/********************************
 * End of Register Instructions *
 ********************************/

/********************
 * LDA Instructions *
 ********************/

		case 0xA9: /* LDA Immediate */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tLDA #$%02X ", IMMEDIATE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A = IMMEDIATE;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 2;
			break;

		case 0xA5: /* LDA ZeroPage */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tLDA $%02X (%02X) ", mem_map[m6502.PC + 1], ZEROPAGE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A = ZEROPAGE;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			/* is it negitive? */
			m6502.P |= N_FLAG & m6502.A;
			if ((mem_map[m6502.PC + 2] == 0xD0) & (mem_map[m6502.PC + 1] == 0x11))
				mem_map[0xD011] = 0x00; /* clear keyboard strobe on read */
			m6502.PC += 2;
			clockcnt += 3;
			break;

		case 0xB5: /* LDA ZeroPage X, (LoaD Acc. from memory) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tLDA $%02X,X (%02X) ", mem_map[m6502.PC + 1], ZEROPAGEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			/* so what happens when X pushes PC past 65535? */
			/* note: treat X as signed */
			m6502.A = ZEROPAGEX;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			/* is it negitive? */
			m6502.P |= N_FLAG & m6502.A;
			m6502.PC += 2;
			clockcnt += 4;
			break;

		case 0xAD: /* LDA Absolute */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tLDA $%02X%02X (%02X) ",
		mem_map[m6502.PC + 2], mem_map[m6502.PC + 1],
		mem_map[(mem_map[m6502.PC + 2] << 8) + mem_map[m6502.PC + 1]]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
#ifdef STOCKREP1
			if (
				(m6502.PC == 0xFF29) |
				(m6502.PC == 0xE003) |
				(m6502.PC == 0xFEC1)) {
#endif
/* for Krus1.3 with new monitor */
			if (	(m6502.PC == 0xE003) | /* WozBasic */
				(m6502.PC == 0xFEEA)) { /* Krus13 wozmon */
/* We are at the wozmon or basic keyboard spinwaits. Block until key available.*/
				if (trace_cpu >= 2)
					fprintf(stderr, "Waiting for a Key\n");
				tmpval = GetKey(); /* this will block until key available */
				if (trace_cpu >= 2)
					fprintf(stderr, "\nGot Key: %02x\n", tmpval);
				mem_map[0xD010] = 0xFF & tmpval;
				m6502.A = 0x80; /* yea, we got a key */
/* Uncomment this next line and the closing endif if you want this to always wait for a key */
/* If you do, it will always block if it looks for a key. */
#ifdef NOTDEF 
			}

			else if ((mem_map[m6502.PC + 2] == 0xD0) & (mem_map[m6502.PC + 1] == 0x11)) {
/*
 * Reading keyboard is a two step process, first see if high bit is set
 * in D011, if so the next key is waiting in D010. We use this first step
 * a blocking point to block wait for the next key
 */
				if (trace_cpu >= 2)
					fprintf(stderr, "Waiting for a Key\n");
				tmpval = GetKey(); /* this will block until key available */
				if (trace_cpu >= 2)
					fprintf(stderr, "\nGot Key: %02x\n", tmpval);
				mem_map[0xD010] = 0xFF & tmpval;
				m6502.A = 0x80; /* yea, we got a key */
#endif /* NOTDEF */
			} else
			m6502.A = mem_map[(mem_map[m6502.PC + 2] << 8) + mem_map[m6502.PC + 1]];

			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			/* is it negitive? */
			m6502.P |= N_FLAG & m6502.A;
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0xBD: /* LDA Absolute X */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tLDA $%02X%02X,X (%04X = %02X) ",
		mem_map[m6502.PC + 2], mem_map[m6502.PC + 1],
		(mem_map[m6502.PC + 2] << 8) + mem_map[m6502.PC + 1] + m6502.X,
		ABSOLUTEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}

			/* if we cross a page boundry, add a clock cycle */
/*			if ((mem_map[m6502.PC + 2] << 8) != (((mem_map[m6502.PC + 2] << 8) + mem_map[m6502.PC + 1] + m6502.X) & 0xFF00))
				clockcnt++; */
			/* so what happens when X pushes PC past 65535? */
			m6502.A = ABSOLUTEX;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			/* is it negitive? */
			m6502.P |= N_FLAG & m6502.A;
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0xB9: /* LDA Absolute Y */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tLDA $%02X%02X,Y (%02X) ",
		mem_map[m6502.PC + 2], mem_map[m6502.PC + 1],
		ABSOLUTEY);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			/* if we cross a page boundry, add a clock cycle */
			/* if ((mem_map[m6502.PC + 2] << 8) != (((mem_map[m6502.PC + 2] << 8) + mem_map[m6502.PC + 1] + m6502.Y) & 0xFF00))
				clockcnt++; */
			/* so what happens when Y pushes PC past 65535? */
			m6502.A = ABSOLUTEY;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			/* is it negitive? */
			m6502.P |= N_FLAG & m6502.A;
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0xA1: /* LDA Indirect X (LoaD Acc. from memory) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tLDA ($%02X,X) (%02X%02X) (%02X) ", mem_map[m6502.PC + 1],
		mem_map[mem_map[m6502.PC + 1] + m6502.X + 1],
		mem_map[mem_map[m6502.PC + 1] + m6502.X],
		INDIRECTX);

	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A = INDIRECTX;
			m6502.P &= ~(N_FLAG | Z_FLAG);
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			/* is it negitive? */
			m6502.P |= N_FLAG & m6502.A;
			m6502.PC += 2;
			clockcnt += 6;
			break;

		case 0xB1: /* LDA Indirect Y (LoaD Acc. from memory) */
			tmpval = mem_map[mem_map[m6502.PC + 1]] + m6502.Y +
				(mem_map[(mem_map[m6502.PC + 1] + 1) & 0xFF] << 8);
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tLDA ($%02X),Y (%04X) (%02X) ", mem_map[m6502.PC + 1],
		tmpval, INDIRECTY);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A = INDIRECTY;
			m6502.P &= ~(N_FLAG | Z_FLAG);
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			/* is it negitive? */
			m6502.P |= N_FLAG & m6502.A;
			if (tmpval == 0xD011)
				mem_map[0xD011] = 0x00; /* clear keyboard strobe on read */
			m6502.PC += 2;
			clockcnt += 5;
			break;

		case 0xB2: /* LDA (ZP) (65C02 Only) */
			tmpval = (mem_map[(mem_map[m6502.PC + 1] + 1) & 0xFF] << 8) +
				mem_map[mem_map[m6502.PC + 1]];
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tLDA ($%02X) (%04X) (%02X) ", mem_map[m6502.PC + 1],
		tmpval, INDIRECTZ);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A = INDIRECTZ;
			m6502.P &= ~(N_FLAG | Z_FLAG);
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			/* is it negitive? */
			m6502.P |= N_FLAG & m6502.A;
			if (tmpval == 0xD011)
				mem_map[0xD011] = 0x00; /* clear keyboard strobe on read */
			m6502.PC += 2;
			clockcnt += 5;
			break;

/***************************
 * End of LDA Instructions *
 ***************************/

/********************
 * LDX Instructions *
 ********************/

		case 0xA2: /* LDX Immediate */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tLDX #$%02X ", IMMEDIATE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.X = IMMEDIATE;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.X) /* is it zero? */
				m6502.P |= Z_FLAG;
			/* is it negitive? */
			m6502.P |= N_FLAG & m6502.X;
			m6502.PC += 2;
			clockcnt += 2;
			break;

		case 0xA6: /* LDX ZeroPage */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tLDX $%02X (%02X) ", mem_map[m6502.PC + 1], ZEROPAGE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.X = ZEROPAGE;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.X) /* is it zero? */
				m6502.P |= Z_FLAG;
			/* is it negitive? */
			m6502.P |= N_FLAG & m6502.X;
			m6502.PC += 2;
			clockcnt += 3;
			break;

		case 0xAE: /* LDX Absolute */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tLDX $%02X%02X (%02X) ",
		mem_map[m6502.PC + 2], mem_map[m6502.PC + 1], ABSOLUTE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if ((mem_map[m6502.PC + 2] == 0xD0) & (mem_map[m6502.PC + 1] == 0x11)) {
/*
 * Reading keyboard is a two step process, first see if high bit is set in D011, if so the next key
 * is waiting in D010. We use this first step a blocking point to block wait for the next key
 */
				if (trace_cpu >= 2)
					fprintf(stderr, "Waiting for a Key\n");
				tmpval = GetKey(); /* this will block until key available */
				if (trace_cpu >= 2)
					fprintf(stderr, "\nGot Key: %02x\n", tmpval);
				mem_map[0xD010] = 0xFF & tmpval;
				m6502.A = 0x80; /* yea, we got a key */
			} else
				m6502.X = ABSOLUTE;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.X) /* is it zero? */
				m6502.P |= Z_FLAG;
			/* is it negitive? */
			m6502.P |= N_FLAG & m6502.X;
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0xBE: /* LDX Absolute Y */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tLDX $%02X%02X,Y (%02X) ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1], ABSOLUTEY);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.X = ABSOLUTEY;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.X) /* is it zero? */
				m6502.P |= Z_FLAG;
			/* is it negitive? */
			m6502.P |= N_FLAG & m6502.X;
			m6502.PC += 3;
			clockcnt += 4;
			break;
		
		case 0xB6: /* LDX Zeropage Y (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tLDX $%02X,Y (%02X) ", mem_map[m6502.PC + 1], ZEROPAGEY);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.X = ZEROPAGEY;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.X) /* is it zero? */
				m6502.P |= Z_FLAG;
			/* is it negitive? */
			m6502.P |= N_FLAG & m6502.X;
			m6502.PC += 2;
			clockcnt += 3;
			break;


/***************************
 * End of LDX Instructions *
 ***************************/

/********************
 * LDY Instructions *
 ********************/

		case 0xA0: /* LDY Immediate */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tLDY #$%02X ", IMMEDIATE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.Y = IMMEDIATE;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.Y) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.Y; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 2;
			break;

		case 0xA4: /* LDY ZeroPage */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tLDY $%02X ($%02X) ", mem_map[m6502.PC + 1], ZEROPAGE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.Y = ZEROPAGE;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.Y) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.Y; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 3;
			break;

		case 0xB4: /* LDY ZeroPage X, (LoaD Y from memory) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tLDY $%02X,X ($%02X) ", mem_map[m6502.PC + 1], ZEROPAGEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.Y = ZEROPAGEX;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.Y) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.Y; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 4;
			break;

		case 0xAC: /* LDY Absolute, (LoaD Y from memory) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tLDY $%02X%02X ($%02X) ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1], ABSOLUTE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.Y = ABSOLUTE;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.Y) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.Y; /* is it negitive? */
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0xBC: /* LDY Absolute X, (LoaD Y from memory) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tLDY $%02X%02X,X ($%02X) ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1], ABSOLUTEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.Y = ABSOLUTEX;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.Y) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.Y; /* is it negitive? */
			m6502.PC += 3;
			clockcnt += 4;
			break;

/***************************
 * End of LDY Instructions *
 ***************************/

/********************
 * STA Instructions *
 ********************/

		case 0x85: /* STA ZeroPage (STore Acc. in memory) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSTA $%02X ", mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			ZEROPAGE = m6502.A;
			m6502.PC += 2;
			clockcnt += 3;
			break;

		case 0x95: /* STA ZeroPage X (STore Acc. in memory) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSTA $%02X,X ", mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			ZEROPAGEX = m6502.A ;
			m6502.PC += 2;
			clockcnt += 4;
			break;

		case 0x8D: /* STA Absolute */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSTA $%02X%02X ", mem_map[m6502.PC + 2],
		mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			ABSOLUTE = m6502.A;
			if ((mem_map[m6502.PC + 2] == 0xD0) & (mem_map[m6502.PC + 1] == 0x12))
				do_output();
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0x9D: /* STA Absolute X */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSTA $%02X%02X,X ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			/* TODO: what happens when X pushes PC past 65535? */
			ABSOLUTEX = m6502.A ;
			m6502.PC += 3;
			clockcnt += 5;
			break;

		case 0x99: /* STA Absolute Y */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSTA $%02X%02X,Y ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			/* TODO: what happens when Y pushes PC past 65535? */
			ABSOLUTEY = m6502.A ;
			m6502.PC += 3;
			clockcnt += 5;
			break;

		case 0x81: /* STA Indirect X (STore Acc. in memory) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSTA ($%02X,X) (%02X%02X) (%02X) ", mem_map[m6502.PC + 1],
		mem_map[mem_map[m6502.PC + 1] + m6502.X + 1],
		mem_map[mem_map[m6502.PC + 1] + m6502.X],
		INDIRECTX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			INDIRECTX = m6502.A;
			m6502.PC += 2;
			clockcnt += 6;
			break;

		case 0x91: /* STA Indirect Y (STore Acc. in memory) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	tmpval = mem_map[mem_map[m6502.PC + 1]] + m6502.Y +
		(mem_map[(mem_map[m6502.PC + 1] + 1) & 0xFF] << 8);
	printf("\tSTA ($%02X),Y (%04X) (%02X) ", mem_map[m6502.PC + 1],
		tmpval, INDIRECTY);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			INDIRECTY = m6502.A;
			m6502.PC += 2;
			clockcnt += 6;
			break;

		case 0x92: /* STA (ZP) (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	tmpval = mem_map[mem_map[m6502.PC + 1]] +
		(mem_map[(mem_map[m6502.PC + 1] + 1) & 0xFF] << 8);
	printf("\tSTA ($%02X) (%04X) (%02X) ", mem_map[m6502.PC + 1],
		tmpval, INDIRECTZ);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			INDIRECTZ = m6502.A;
			m6502.PC += 2;
			clockcnt += 5;
			break;

/***************************
 * End of STA Instructions *
 ***************************/

/********************
 * CMP Instructions *
 ********************/

		case 0xC9: /* CMP Immediate (CoMPare accumulator) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tCMP #$%02X ", IMMEDIATE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(C_FLAG | N_FLAG | Z_FLAG);
			/* if A is > value, set C_FLAG */
			if (m6502.A > IMMEDIATE)
				m6502.P |= C_FLAG;
			/* if A is == value, set Z_FLAG and the C_FLAG */
			else if (m6502.A == IMMEDIATE)
				m6502.P |= C_FLAG | Z_FLAG;
			/* if A - value results in negitive, set N_FLAG */
			m6502.P |= N_FLAG & (m6502.A - IMMEDIATE);
			m6502.PC += 2;
			clockcnt += 2;
			break;

		case 0xC5: /* CMP ZeroPage */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tCMP $%02X (%02X) ", mem_map[m6502.PC + 1], ZEROPAGE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(C_FLAG | N_FLAG | Z_FLAG);
			/* if A is > value, set C_FLAG */
			if (m6502.A > ZEROPAGE)
				m6502.P |= C_FLAG;
			/* if A is == value, set Z_FLAG */
			else if (m6502.A == ZEROPAGE)
				m6502.P |= Z_FLAG | C_FLAG;
			/* if A - value results in negitive, set N_FLAG */
			m6502.P |= N_FLAG & (m6502.A - ZEROPAGE);
			m6502.PC += 2;
			clockcnt += 3;
			break;

		case 0xD5: /* CMP ZeroPage X, (CoMPare memory with acc.) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tCMP $%02X,X (%02X) ", mem_map[m6502.PC + 1], ZEROPAGEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(C_FLAG | N_FLAG | Z_FLAG);
			/* if A is > value, set C_FLAG */
			if (m6502.A > ZEROPAGEX)
				m6502.P |= C_FLAG;
			/* if A is == value, set Z_FLAG and C_FLAG */
			else if (m6502.A == ZEROPAGEX)
				m6502.P |= Z_FLAG | C_FLAG;
			/* if A - value results in negitive, set N_FLAG */
			m6502.P |= N_FLAG & (m6502.A - ZEROPAGEX);
			m6502.PC += 2;
			clockcnt += 4;
			break;

		case 0xCD: /* CMP Absolute */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tCMP $%02X%02X (%02X) ", mem_map[m6502.PC + 2],
		mem_map[m6502.PC + 1], ABSOLUTE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(C_FLAG | N_FLAG | Z_FLAG);
			/* if A is > value, set C_FLAG */
			if (m6502.A > ABSOLUTE)
				m6502.P |= C_FLAG;
			/* if A is == value, set Z_FLAG and C_FLAG */
			else if (m6502.A == ABSOLUTE)
				m6502.P |= Z_FLAG | C_FLAG;
			/* if A - value results in negitive, set N_FLAG */
			m6502.P |= N_FLAG & (m6502.A - ABSOLUTE);
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0xDD: /* CMP Absolute X */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tCMP $%02X%02X,X (%02X) ", mem_map[m6502.PC + 2],
		mem_map[m6502.PC + 1], ABSOLUTEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(C_FLAG | N_FLAG | Z_FLAG);
			/* if A is > value, set C_FLAG */
			if (m6502.A > ABSOLUTEX)
				m6502.P |= C_FLAG;
			else if (m6502.A == ABSOLUTEX) {
			/* if A is == value, set Z_FLAG and C_FLAG */
				m6502.P |= Z_FLAG | C_FLAG;
			}
			/* if A - value results in negitive, set N_FLAG */
				m6502.P |= N_FLAG & (m6502.A - ABSOLUTEX);
			
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0xD9: /* CMP Absolute Y */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tCMP $%02X%02X,Y (%02X) ", mem_map[m6502.PC + 2],
		mem_map[m6502.PC + 1], ABSOLUTEY);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(C_FLAG | N_FLAG | Z_FLAG);
			/* if A is > value, set C_FLAG */
			if (m6502.A > ABSOLUTEY)
				m6502.P |= C_FLAG;
			/* if A is == value, set Z_FLAG and C_FLAG */
			else if (m6502.A == ABSOLUTEY)
				m6502.P |= Z_FLAG | C_FLAG;
			/* if A - value results in negitive, set N_FLAG */
			m6502.P |= N_FLAG & (m6502.A - ABSOLUTEY);
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0xC1: /* CMP Indirect X (CoMPare memory and acc.) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tCMP ($%02X,X) (%02X%02X) (%02X) ", mem_map[m6502.PC + 1],
		mem_map[mem_map[m6502.PC + 1] + m6502.X + 1],
		mem_map[mem_map[m6502.PC + 1] + m6502.X],
		INDIRECTX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(C_FLAG | N_FLAG | Z_FLAG);
			/* if A is > value, set C_FLAG */
			if (m6502.A > INDIRECTX)
				m6502.P |= C_FLAG;
			/* if A is == value, set Z_FLAG and C_FLAG */
			else if (m6502.A == INDIRECTX)
				m6502.P |= Z_FLAG | C_FLAG;
			/* if A - value results in negitive, set N_FLAG */
			m6502.P |= N_FLAG & (m6502.A - INDIRECTX);
			m6502.PC += 2;
			clockcnt += 6;
			break;

		case 0xD1: /* CMP Indirect Y (CoMPare memory and acc.) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	tmpval = mem_map[mem_map[m6502.PC + 1]] + m6502.Y +
		(mem_map[(mem_map[m6502.PC + 1] + 1) & 0xFF] << 8);
	printf("\tCMP ($%02X),Y (%04X) (%02X) ", mem_map[m6502.PC + 1],
		tmpval, INDIRECTY);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(C_FLAG | N_FLAG | Z_FLAG);
			/* if A is > value, set C_FLAG */
			if (m6502.A > INDIRECTY)
				m6502.P |= C_FLAG;
			/* if A is == value, set Z_FLAG */
			else if (m6502.A == INDIRECTY)
				m6502.P |= Z_FLAG | C_FLAG;
			/* if A - value results in negitive, set N_FLAG */
			m6502.P |= N_FLAG & (m6502.A - INDIRECTY);
			m6502.PC += 2;
			clockcnt += 5;
			break;

		case 0xD2: /* CMP (ZP) (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	tmpval = mem_map[mem_map[m6502.PC + 1]] +
		(mem_map[(mem_map[m6502.PC + 1] + 1) & 0xFF] << 8);
	printf("\tCMP ($%02X) (%04X) (%02X) ", mem_map[m6502.PC + 1],
		tmpval, INDIRECTZ);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(C_FLAG | N_FLAG | Z_FLAG);
			/* if A is > value, set C_FLAG */
			if (m6502.A > INDIRECTZ)
				m6502.P |= C_FLAG;
			/* if A is == value, set Z_FLAG */
			else if (m6502.A == INDIRECTZ)
				m6502.P |= Z_FLAG | C_FLAG;
			/* if A - value results in negitive, set N_FLAG */
			m6502.P |= N_FLAG & (m6502.A - INDIRECTZ);
			m6502.PC += 2;
			clockcnt += 5;
			break;

/***************************
 * End of CMP Instructions *
 ***************************/

/********************
 * CPX Instructions *
 ********************/

		case 0xE0: /* CPX Immediate */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tCPX #$%02X ", IMMEDIATE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(C_FLAG | N_FLAG | Z_FLAG);
			/* if A is > value, set C_FLAG */
			if (m6502.X > IMMEDIATE)
				m6502.P |= C_FLAG;
			/* if A is == value, set Z_FLAG and C_FLAG */
			else if (m6502.X == IMMEDIATE)
				m6502.P |= Z_FLAG | C_FLAG;
			/* if A - value results in negitive, set N_FLAG */
			m6502.P |= N_FLAG & (m6502.X - IMMEDIATE);
			m6502.PC += 2;
			clockcnt += 2;
			break;

		case 0xE4: /* CPX ZeroPage */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tCPX $%02X (%02X) ", mem_map[m6502.PC + 1], ZEROPAGE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(C_FLAG | N_FLAG | Z_FLAG);
			/* if A is > value, set C_FLAG */
			if (m6502.X > ZEROPAGE)
				m6502.P |= C_FLAG;
			/* if X is == value, set Z_FLAG and C_FLAG */
			else if (m6502.X == ZEROPAGE)
				m6502.P |= Z_FLAG | C_FLAG;
			/* if A - value results in negitive, set N_FLAG */
			m6502.P |= N_FLAG & (m6502.X - ZEROPAGE);
			m6502.PC += 2;
			clockcnt += 3;
			break;

		case 0xEC: /* CPX Absolute */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tCPX $%02X%02X (%02X) ", mem_map[m6502.PC + 2],
		mem_map[m6502.PC + 1], ABSOLUTE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(C_FLAG | N_FLAG | Z_FLAG);
			/* if X is > value, set C_FLAG */
			if (m6502.X > ABSOLUTE)
				m6502.P |= C_FLAG;
			/* if X is == value, set Z_FLAG and C_FLAG */
			if (m6502.X == ABSOLUTE)
				m6502.P |= Z_FLAG | C_FLAG;
			/* if A - value results in negitive, set N_FLAG */
			m6502.P |= N_FLAG & (m6502.X - ABSOLUTE);
			m6502.PC += 3;
			clockcnt += 4;
			break;

/***************************
 * End of CPX Instructions *
 ***************************/

/********************
 * CPY Instructions *
 ********************/

		case 0xC0: /* CPY Immediate */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tCPY #$%02X ", IMMEDIATE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(C_FLAG | Z_FLAG | N_FLAG);
			/* if Y is > value, set C_FLAG */
			if (m6502.Y > IMMEDIATE)
				m6502.P |= C_FLAG;
			/* if Y is == value, set Z_FLAG and C_FLAG */
			else if (m6502.Y == IMMEDIATE)
				m6502.P |= Z_FLAG | C_FLAG;
			/* if Y - value results in negitive, set N_FLAG */
			if ((m6502.Y - IMMEDIATE) & 0x80)
				m6502.P |= N_FLAG;
			m6502.PC += 2;
			clockcnt += 2;
			break;

		case 0xC4: /* CPY ZeroPage */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tCPY $%02X (%02X) ", mem_map[m6502.PC + 1], ZEROPAGE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(C_FLAG | Z_FLAG | N_FLAG);
			/* if Y is > value, set C_FLAG */
			if (m6502.Y > ZEROPAGE)
				m6502.P |= C_FLAG;
			/* if Y is == value, set Z_FLAG and C_FLAG */
			else if (m6502.Y == ZEROPAGE)
				m6502.P |= Z_FLAG | C_FLAG;
			/* if Y - value results in negitive, set N_FLAG */
			m6502.P |= N_FLAG & (m6502.Y - ZEROPAGE);
			m6502.PC += 2;
			clockcnt += 3;
			break;

		case 0xCC: /* CPY Absolute */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tCPY $%02X%02X (%02X) ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1], ABSOLUTE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(C_FLAG | Z_FLAG | N_FLAG);
			/* if Y is > value, set C_FLAG */
			if (m6502.Y > ABSOLUTE)
				m6502.P |= C_FLAG;
			/* if Y is == value, set Z_FLAG and C_FLAG */
			else if (m6502.Y == ABSOLUTE)
				m6502.P |= Z_FLAG | C_FLAG;
			/* if Y - value results in negitive, set N_FLAG */
			m6502.P |= N_FLAG & (m6502.Y - ABSOLUTE);
			m6502.PC += 3;
			clockcnt += 4;
			break;

/***************************
 * End of CPY Instructions *
 ***************************/

/**********************
 * Stack Instructions *
 **********************/

		case 0x9A: /* TXS (transfer X to SP) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tTXS ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.SP = m6502.X;
			m6502.PC += 1;
			clockcnt += 2;
			break;

		case 0xBA: /* TSX (transfer SP to X) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tTSX ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.X = m6502.SP;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.X) /* is it zero? */
				m6502.P |= Z_FLAG;
			/* is it negitive? */
			m6502.P |= N_FLAG & m6502.X;
			m6502.PC += 1;
			clockcnt += 2;
			break;

		case 0x48: /* PHA (push A on the stack) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tPHA ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			mem_map[0x100 + m6502.SP--] = m6502.A;
			m6502.PC += 1;
			clockcnt += 3;
			break;

		case 0x68: /* PLA (pull A from stack) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tPLA ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A = mem_map[0x100 + ++m6502.SP];
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			/* is it negitive? */
			m6502.P |= N_FLAG & m6502.A;
			m6502.PC += 1;
			clockcnt += 4;
			break;

		case 0x08: /* PHP (push P on the stack) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tPHP ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			mem_map[0x100 + m6502.SP--] = m6502.P;
			m6502.PC += 1;
			clockcnt += 3;
			break;

		case 0x28: /* PLP (pull P from stack) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tPLP ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P = mem_map[0x100 + ++m6502.SP];
			m6502.PC += 1;
			clockcnt += 4;
			break;

/*****************************
 * End of Stack Instructions *
 *****************************/

/********************
 * JMP Instructions *
 ********************/

/*
JMP transfers program execution to the following address (absolute) or to the location contained in the following address (indirect). Note that there is no carry associated with the indirect jump so:
AN INDIRECT JUMP MUST NEVER USE A VECTOR BEGINNING ON THE LAST BYTE OF A PAGE
For example if address $3000 contains $40, $30FF contains $80, and $3100 contains $50, the result of JMP ($30FF) will be a transfer of control to $4080 rather than $5080 as you intended i.e. the 6502 took the low byte of the address from $30FF and the high byte from $3000. */

		case 0x4C: /* JMP Absolute */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tJMP $%02X%02X ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.PC = (mem_map[m6502.PC + 2] << 8) + mem_map[m6502.PC + 1];
			clockcnt += 3;
			break;

		case 0x6C: /* JMP Indirect */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tJMP ($%02X%02X) ((%02X%02X)) ",
		mem_map[m6502.PC + 2], mem_map[m6502.PC + 1],
		mem_map[(mem_map[m6502.PC + 2] << 8) + mem_map[m6502.PC + 1] + 1],
		mem_map[(mem_map[m6502.PC + 2] << 8) + mem_map[m6502.PC + 1]]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.PC = (mem_map[(mem_map[m6502.PC + 2] << 8) +
				mem_map[m6502.PC + 1] + 1] << 8) +
				mem_map[(mem_map[m6502.PC + 2] << 8) +
				mem_map[m6502.PC + 1]];
			clockcnt += 3;
			break;

		case 0x7C: /* JMP Absolute Indexed Indirect (65C02 only)  */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tJMP ($%02X%02X,X) ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			tmpval = (mem_map[m6502.PC + 2] << 8) + mem_map[m6502.PC + 1] + m6502.X;
			m6502.PC = (mem_map[tmpval + 1] << 8) + mem_map[tmpval];
			clockcnt += 6;
			break;

/***************************
 * End of JMP Instructions *
 ***************************/

/**************************
 * Interrupt Instructions *
 **************************/

		case 0x00: /* BRK */
/* BRK causes a non-maskable interrupt and increments the program counter by two. Therefore an RTI will go to the address of the BRK +3 so that BRK may be used to replace a three-byte instruction for debugging and the subsequent RTI will be correct. */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tBRK ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			mem_map[0x100 + m6502.SP--] = (2 + m6502.PC) >> 8;
			mem_map[0x100 + m6502.SP--] = (2 + m6502.PC) & 0xFF;
			m6502.P |= B_FLAG;
			m6502.P &= ~D_FLAG;
			/* m6502.P |= I_FLAG; */
			mem_map[0x100 + m6502.SP--] = m6502.P;
			m6502.PC = (mem_map[0xFFFF] << 8) + mem_map[0xFFFE];
			clockcnt += 7;
			break;

		case 0x40: /* RTI */
/* RTI retrieves the Processor Status Word (flags) and the Program Counter from the stack in that order (interrupts push the PC first and then the PSW).
Note that unlike RTS, the return address on the stack is the actual address rather than the address-1.*/
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tRTI ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P = mem_map[0x101 + m6502.SP];
			m6502.PC = (mem_map[0x103 + m6502.SP] << 8) + mem_map[0x102 + m6502.SP];
			m6502.SP += 3;
			clockcnt += 6;
			break;

		case 0x20: /* JSR */
/* JSR pushes the address-1 of the next operation on to the stack before transferring program control to the following address. Subroutines are normally terminated by a RTS op code. */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tJSR $%02X%02X ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			mem_map[0x100 + m6502.SP--] = (m6502.PC + 2) >> 8;
			mem_map[0x100 + m6502.SP--] = (m6502.PC + 2) & 0xFF;
			m6502.PC = (mem_map[m6502.PC + 2] << 8) + mem_map[m6502.PC + 1];
			clockcnt += 5;
			break;

		case 0x60: /* RTS */
/*RTS pulls the top two bytes off the stack (low byte first) and transfers program control to that address+1. It is used, as expected, to exit a subroutine invoked via JSR which pushed the address-1.*/
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tRTS $%02X%02X ", mem_map[0x102 + m6502.SP],
		mem_map[0x101 + m6502.SP] + 1);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.PC = mem_map[0x100 + ++m6502.SP];
			m6502.PC += mem_map[0x100 + ++m6502.SP] << 8;
			m6502.PC++;
			clockcnt += 6;
			break;


/*********************************
 * End of Interrupt Instructions *
 *********************************/


/********************
 * BIT Instructions *
 ********************/

		case 0x24: /* BIT ZeroPage */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tBIT $%02X (%02X)", mem_map[m6502.PC + 1], ZEROPAGE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(N_FLAG | V_FLAG | Z_FLAG);
			m6502.P |= ZEROPAGE & (N_FLAG | V_FLAG);
			if (!(ZEROPAGE & m6502.A))
				m6502.P |= Z_FLAG;
			m6502.PC += 2;
			clockcnt += 3;
			break;

		case 0x34: /* BIT ZeroPage X (BIT test) (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tBIT $%02X,X (%02X)", mem_map[m6502.PC + 1], ZEROPAGEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(N_FLAG | V_FLAG | Z_FLAG);
			m6502.P |= ZEROPAGEX & (N_FLAG | V_FLAG);
			if (!(ZEROPAGEX & m6502.A))
				m6502.P |= Z_FLAG;
			m6502.PC += 2;
			clockcnt += 3;
			break;

		case 0x2C: /* BIT Absolute */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tBIT $%02X%02X (%02X) ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1], ABSOLUTE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(N_FLAG | V_FLAG | Z_FLAG);
			m6502.P |= ABSOLUTE & (N_FLAG | V_FLAG);
			if (!(ABSOLUTE & m6502.A))
				m6502.P |= Z_FLAG;
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0x3C: /* BIT Absolute X (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tBIT $%02X%02X,X (%02X) ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1], ABSOLUTEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(N_FLAG | V_FLAG | Z_FLAG);
			m6502.P |= ABSOLUTEX & (N_FLAG | V_FLAG);
			if (!(ABSOLUTEX & m6502.A))
				m6502.P |= Z_FLAG;
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0x89: /* BIT Immediate (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tBIT #$%02X", IMMEDIATE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(N_FLAG | V_FLAG | Z_FLAG);
			tmpval = IMMEDIATE;
			m6502.P |= tmpval & (N_FLAG | V_FLAG);
			if (!(tmpval & m6502.A))
				m6502.P |= Z_FLAG;
			m6502.PC += 2;
			clockcnt += 3;
			break;


/***************************
 * End of BIT Instructions *
 ***************************/

/********************
 * ASL Instructions *
 ********************/

		case 0x0A: /* ASL Accumulator, Arithmetic Shift Left one bit */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tASL A ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(C_FLAG | Z_FLAG | N_FLAG);
			if (m6502.A & 0x80) /* is the high bit set? */
				m6502.P |= C_FLAG; /* set the carry */
			m6502.A = m6502.A << 1; /* shift it left */
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 1;
			clockcnt += 2;
			break;

		case 0x06: /* ASL ZeroPage, Arthmetic Shift Left one bit */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tASL $%02X (%02X)", mem_map[m6502.PC + 1], ZEROPAGE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(C_FLAG | Z_FLAG | N_FLAG);
			if (ZEROPAGE & 0x80) /* is the high bit set? */
				m6502.P |= C_FLAG;
			/* shift it left */
			ZEROPAGE = ZEROPAGE << 1;
			if (!ZEROPAGE) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & ZEROPAGE; /* is it now negitive? */
			m6502.PC += 2;
			clockcnt += 5;
			break;

		case 0x16: /* ASL ZeroPage X, (Arthmetic Shift Left one bit) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tASL $%02X,X (%02X)", mem_map[m6502.PC + 1], ZEROPAGEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(C_FLAG | Z_FLAG | N_FLAG);
			if (ZEROPAGEX & 0x80) /* is the high bit set? */
				m6502.P |= C_FLAG;
			/* shift it left */
			ZEROPAGEX = ZEROPAGEX << 1;
			m6502.A = m6502.A << 1;
			if (!ZEROPAGEX) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & ZEROPAGEX; /* is it now negitive? */
			m6502.PC += 2;
			clockcnt += 6;
			break;

		case 0x0E: /* ASL Absolute, (Arthmetic Shift Left one bit) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tASL $%02X%02X (%02X)", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1], ABSOLUTE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(C_FLAG | Z_FLAG | N_FLAG);
			if (ABSOLUTE & 0x80) /* is the high bit set? */
				m6502.P |= C_FLAG;
			/* shift it left */
			ABSOLUTE = ABSOLUTE << 1;
			m6502.A = m6502.A << 1;
			if (!ABSOLUTE) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & ZEROPAGEX; /* is it now negitive? */
			m6502.PC += 3;
			clockcnt += 6;
			break;

		case 0x1E: /* ASL Absolute X, (Arthmetic Shift Left one bit) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tASL $%02X%02X,X (%02X)", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1], ABSOLUTEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(C_FLAG | Z_FLAG | N_FLAG);
			if (ABSOLUTEX & 0x80) /* is the high bit set? */
				m6502.P |= C_FLAG;
			/* shift it left */
			ABSOLUTEX = ABSOLUTEX << 1;
			m6502.A = m6502.A << 1;
			if (!ABSOLUTEX) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & ZEROPAGEX; /* is it now negitive? */
			m6502.PC += 3;
			clockcnt += 7;
			break;

/***************************
 * End of ASL Instructions *
 ***************************/

/********************
 * STY Instructions *
 ********************/

		case 0x84: /* STY ZeroPage, store Y at $00XX */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSTY $%02X ", mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			ZEROPAGE = m6502.Y;
			m6502.PC += 2;
			clockcnt += 3;
			break;

		case 0x94: /* STY ZeroPage X , (STore Y in memory) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSTY $%02X,X ", mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			ZEROPAGEX = m6502.Y;
			m6502.PC += 2;
			clockcnt += 4;
			break;

		case 0x8C: /* STY Absolute */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSTY $%02X%02X ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			ABSOLUTE = m6502.Y;
			m6502.PC += 3;
			clockcnt += 4;
			break;

/***************************
 * End of STY Instructions *
 ***************************/

/********************
 * STX Instructions *
 ********************/

		case 0x86: /* STX ZeroPage, store X at $00XX */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSTX $%02X", mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			ZEROPAGE = m6502.X;
			m6502.PC += 2;
			clockcnt += 3;
			break;

		case 0x96: /* STX ZeroPage,Y store X at $00XX + Y */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSTX $%02X,Y", mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			ZEROPAGEY = m6502.X;
			m6502.PC += 2;
			clockcnt += 4;
			break;

		case 0x8E: /* STX Absolute, store X at $YYXX */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSTX $%02X%02X", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			ABSOLUTE = m6502.X;
			m6502.PC += 3;
			clockcnt += 4;
			break;

/***************************
 * End of STX Instructions *
 ***************************/

/********************
 * EOR Instructions *
 ********************/

		case 0x49: /* EOR Immediate (Exclusive OR acc. with memory) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tEOR #$%02X", IMMEDIATE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A ^= IMMEDIATE;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 2;
			break;

		case 0x45: /* EOR ZeroPage (Exclusive OR acc. with memory) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tEOR $%02X (%02X)", mem_map[m6502.PC + 1], ZEROPAGE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A ^= ZEROPAGE;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 3;
			break;

		case 0x55: /* EOR ZeroPage X (Exclusive OR acc. with memory) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tEOR $%02X,X (%02X)", mem_map[m6502.PC + 1], ZEROPAGEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A ^= ZEROPAGEX;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 4;
			break;

		case 0x4D: /* EOR Absolute (Exclusive OR acc. with memory) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tEOR $%02X%02X (%02X)", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1],
		ABSOLUTE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A ^= ABSOLUTE;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0x5D: /* EOR Absolute X (Exclusive OR acc. with memory) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tEOR $%02X%02X,X (%02X)", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1],
		ABSOLUTEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A ^= ABSOLUTEX;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0x59: /* EOR Absolute Y (Exclusive OR acc. with memory) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tEOR $%02X%02X,Y (%02X)", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1],
		ABSOLUTEY);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A ^= ABSOLUTEY;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0x41: /* EOR Indirect X (Exclusive OR mem with acc.) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tEOR ($%02X,X) (%02X%02X) (%02X) ", mem_map[m6502.PC + 1],
		mem_map[mem_map[m6502.PC + 1] + m6502.X + 1],
		mem_map[mem_map[m6502.PC + 1] + m6502.X],
		INDIRECTX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A ^= INDIRECTX;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 6;
			break;

		case 0x51: /* EOR Indirect Y (Exclusive OR mem with acc.) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	tmpval = mem_map[mem_map[m6502.PC + 1]] + m6502.Y +
		(mem_map[(mem_map[m6502.PC + 1] + 1) & 0xFF] << 8);
	printf("\tEOR ($%02X),Y (%04X) (%02X) ", mem_map[m6502.PC + 1],
		tmpval, INDIRECTY);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A ^= INDIRECTY;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 5;
			break;

		case 0x52: /* EOR (ZP) (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	tmpval = mem_map[mem_map[m6502.PC + 1]] +
		(mem_map[(mem_map[m6502.PC + 1] + 1) & 0xFF] << 8);
	printf("\tEOR ($%02X) (%04X) (%02X) ", mem_map[m6502.PC + 1],
		tmpval, INDIRECTZ);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A ^= INDIRECTZ;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 5;
			break;

/***************************
 * End of EOR Instructions *
 ***************************/

/********************
 * ADC Instructions *
 ********************/

		case 0x69: /* ADC Immediate (Add with Carry) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tADC #$%02X ", IMMEDIATE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (m6502.P & D_FLAG) { /* is it decimal? */
				printf("\tADC (0x69) BCD not supported yet\n");
				exit (1);
			}
			tmpval = m6502.A + IMMEDIATE;
			if (m6502.P & C_FLAG) tmpval++;
			m6502.P &= ~(N_FLAG | C_FLAG | V_FLAG | Z_FLAG);
			/* test for carry */
			if (tmpval & 0x100)
				m6502.P |= C_FLAG;
			/* test for overflow */
			if (!((m6502.A ^ tmpval) & 0x80) && ((m6502.A ^ tmpval) & 0x80))
				m6502.P |= V_FLAG;
			m6502.A = tmpval;
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 2;
			break;

		case 0x65: /* ADC ZeroPage (Add with Carry) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tADC $%02X ($%02X) ", mem_map[m6502.PC + 1], ZEROPAGE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (m6502.P & D_FLAG) { /* is it decimal? */
				printf("\tADC (0x65) BCD not supported yet\n");
				exit (1);
			}
			tmpval = m6502.A + ZEROPAGE + ((m6502.P & C_FLAG) ? 1 : 0);
			m6502.P &= ~(N_FLAG | C_FLAG | V_FLAG | Z_FLAG);
			/* test for carry */
			if (tmpval & 0x100)
				m6502.P |= C_FLAG;
			/* test for overflow */
			if (!((m6502.A ^ tmpval) & 0x80) && ((m6502.A ^ tmpval) & 0x80))
				m6502.P |= V_FLAG;
			m6502.A = tmpval;
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 3;
			break;

		case 0x75: /* ADC ZeroPage X (Add mem with Carry) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tADC $%02X,X ($%02X)", mem_map[m6502.PC + 1], ZEROPAGEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (m6502.P & D_FLAG) { /* is it decimal? */
				printf("\tADC (0x75) BCD not supported yet\n");
				exit (1);
			}
			tmpval = m6502.A + ZEROPAGEX + ((m6502.P & C_FLAG) ? 1 : 0);
			m6502.P &= ~(N_FLAG | C_FLAG | V_FLAG | Z_FLAG);
			if (tmpval & 0x100) /* test for carry */
				m6502.P |= C_FLAG;
			/* test for overflow */
			if (!((m6502.A ^ tmpval) & 0x80) && ((m6502.A ^ tmpval) & 0x80))
				m6502.P |= V_FLAG;
			m6502.A = tmpval;
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 4;
			break;

		case 0x6D: /* ADC Absolute (Add with Carry) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tADC $%02X%02X ($%02X) ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1], ABSOLUTE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (m6502.P & D_FLAG) { /* is it decimal? */
				printf("\tADC (0x6D) BCD not supported yet\n");
				exit (1);
			}
			tmpval = m6502.A + ABSOLUTE + ((m6502.P & C_FLAG) ? 1 : 0);
			m6502.P &= ~(N_FLAG | C_FLAG | V_FLAG | Z_FLAG);
			/* test for carry */
			if (tmpval & 0x100)
				m6502.P |= C_FLAG;
			/* test for overflow */
			if (!((m6502.A ^ tmpval) & 0x80) && ((m6502.A ^ tmpval) & 0x80))
				m6502.P |= V_FLAG;
			m6502.A = tmpval;
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0x7D: /* ADC Absolute X (Add with Carry) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tADC $%02X%02X,X ($%02X) ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1],
		ABSOLUTEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (m6502.P & D_FLAG) { /* is it decimal? */
				printf("\tADC (0x7D) BCD not supported yet\n");
				exit (1);
			}
			tmpval = m6502.A + ABSOLUTEX + ((m6502.P & C_FLAG) ? 1 : 0);
			m6502.P &= ~(N_FLAG | C_FLAG | V_FLAG | Z_FLAG);
			/* test for carry */
			if (tmpval & 0x100)
				m6502.P |= C_FLAG;
			/* test for overflow */
			if (!((m6502.A ^ tmpval) & 0x80) && ((m6502.A ^ tmpval) & 0x80))
				m6502.P |= V_FLAG;
			m6502.A = tmpval;
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0x79: /* ADC Absolute Y (Add with Carry) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tADC $%02X%02X,Y ($%02X) ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1],
		ABSOLUTEY);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (m6502.P & D_FLAG) { /* is it decimal? */
				printf("\tADC (0x79) BCD not supported yet\n");
				exit (1);
			}
			tmpval = m6502.A + ABSOLUTEY + ((m6502.P & C_FLAG) ? 1 : 0);
			m6502.P &= ~(N_FLAG | C_FLAG | V_FLAG | Z_FLAG);
			/* test for carry */
			if (tmpval & 0x100)
				m6502.P |= C_FLAG;
			/* test for overflow */
			if (!((m6502.A ^ tmpval) & 0x80) && ((m6502.A ^ tmpval) & 0x80))
				m6502.P |= V_FLAG;
			m6502.A = tmpval;
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0x61: /* ADC Indirect X (ADd mem to acc. with Carry) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tADC ($%02X,X) (%02X%02X) (%02X) ", mem_map[m6502.PC + 1],
		mem_map[mem_map[m6502.PC + 1] + m6502.X + 1],
		mem_map[mem_map[m6502.PC + 1] + m6502.X],
		INDIRECTX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (m6502.P & D_FLAG) { /* is it decimal? */
				printf("\tADC (0x61) BCD not supported yet\n");
				exit (1);
			}
			tmpval = m6502.A + INDIRECTX;
			if (m6502.P & C_FLAG) tmpval++;
			m6502.P &= ~(N_FLAG | C_FLAG | V_FLAG | Z_FLAG);
			/* test for carry */
			if (tmpval & 0x100)
				m6502.P |= C_FLAG;
			/* test for overflow */
			if (!((m6502.A ^ tmpval) & 0x80) && ((m6502.A ^ tmpval) & 0x80))
				m6502.P |= V_FLAG;
			m6502.A = tmpval;
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 6;
			break;

		case 0x71: /* ADC Indirect Y (Add with Carry) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	tmpval = mem_map[mem_map[m6502.PC + 1]] + m6502.Y +
		(mem_map[(mem_map[m6502.PC + 1] + 1) & 0xFF] << 8);
	printf("\tADC ($%02X),Y (%04X) (%02X) ", mem_map[m6502.PC + 1],
		tmpval, INDIRECTY);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (m6502.P & D_FLAG) { /* is it decimal? */
				printf("\tADC (0x71) BCD not supported yet\n");
				exit (1);
			}
			tmpval = m6502.A + INDIRECTY + ((m6502.P & C_FLAG) ? 1 : 0);
			m6502.P &= ~(N_FLAG | C_FLAG | V_FLAG | Z_FLAG);
			/* test for carry */
			if (tmpval & 0x100)
				m6502.P |= C_FLAG;
			/* test for overflow */
			if (!((m6502.A ^ tmpval) & 0x80) && ((m6502.A ^ tmpval) & 0x80))
				m6502.P |= V_FLAG;
			m6502.A = tmpval;
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 5;
			break;

		case 0x72: /* ADC (ZP) (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	tmpval = mem_map[mem_map[m6502.PC + 1]] +
		(mem_map[(mem_map[m6502.PC + 1] + 1) & 0xFF] << 8);
	printf("\tADC ($%02X) (%04X) (%02X) ", mem_map[m6502.PC + 1],
		tmpval, INDIRECTZ);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (m6502.P & D_FLAG) { /* is it decimal? */
				printf("\tADC (0x72) BCD not supported yet\n");
				exit (1);
			}
			tmpval = m6502.A + INDIRECTZ + ((m6502.P & C_FLAG) ? 1 : 0);
			m6502.P &= ~(N_FLAG | C_FLAG | V_FLAG | Z_FLAG);
			/* test for carry */
			if (tmpval & 0x100)
				m6502.P |= C_FLAG;
			/* test for overflow */
			if (!((m6502.A ^ tmpval) & 0x80) && ((m6502.A ^ tmpval) & 0x80))
				m6502.P |= V_FLAG;
			m6502.A = tmpval;
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 5;
			break;

/***************************
 * End of ADC Instructions *
 ***************************/

/********************
 * SBC Instructions *
 ********************/

		case 0xE9: /* SBC Immediate (SuBtract with Carry) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSBC #$%02X ", IMMEDIATE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (m6502.P & D_FLAG) { /* is it decimal? */
				printf("\tSBC (0xE9) BCD not supported yet\n");
				exit (1);
			}
			tmpval = m6502.A;
			if (m6502.P & C_FLAG)
				tmpval++;
			tmpval += IMMEDIATE ^ 0xFF; /* ^ xor bits */
			m6502.P &= ~(N_FLAG | C_FLAG | V_FLAG | Z_FLAG);
			/* test for carry */
			if (tmpval & 0x100)
				m6502.P |= C_FLAG;
			tmpval &= 0xFF;
			/* test for overflow */
			if ((m6502.A ^ IMMEDIATE) &
				(m6502.A ^ tmpval) & 0x80)
				m6502.P |= V_FLAG;
			m6502.A = tmpval;
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 2;
			break;

		case 0xE5: /* SBC ZeroPage (Subtract with Carry) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSBC $%02X (%02X) ", mem_map[m6502.PC + 1], ZEROPAGE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (m6502.P & D_FLAG) { /* is it decimal? */
				printf("\tSBC (0xE5) BCD not supported yet\n");
				exit (1);
			}
			tmpval = m6502.A;
			if (m6502.P & C_FLAG)
				tmpval++;
			tmpval += ZEROPAGE ^ 0xFF;
			m6502.P &= ~(N_FLAG | C_FLAG | V_FLAG | Z_FLAG);
			/* test for carry */
			if (tmpval & 0x100)
				m6502.P |= C_FLAG;
			tmpval &= 0xFF;
			/* test for overflow */
			if ((m6502.A ^ ZEROPAGE) &
				(m6502.A ^ tmpval) & 0x80)
				m6502.P |= V_FLAG;
			m6502.A = tmpval;
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 3;
			break;

		case 0xF5: /* SBC ZeroPage X (Subtract mem from acc. with Carry) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSBC ($%02X),X (%02X) ", mem_map[m6502.PC + 1], ZEROPAGEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}

			if (m6502.P & D_FLAG) { /* is it decimal? */
				printf("\tSBC (0xF5) BCD not supported yet\n");
				exit (1);
			}
			tmpval = m6502.A;
			if (m6502.P & C_FLAG)
				tmpval++;
			tmpval += ZEROPAGEX ^ 0xFF;
			m6502.P &= ~(N_FLAG | C_FLAG | V_FLAG | Z_FLAG);
			/* test for carry */
			if (tmpval & 0x100)
				m6502.P |= C_FLAG;
			tmpval &= 0xFF;
			/* test for overflow */
			if ((m6502.A ^ ZEROPAGEX) & (m6502.A ^ tmpval) & 0x80)
				m6502.P |= V_FLAG;
			m6502.A = tmpval;
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 4;
			break;

		case 0xED: /* SBC Absolute (Subtract with Carry) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSBC $%02X%02X (%02X) ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1], ABSOLUTE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (m6502.P & D_FLAG) { /* is it decimal? */
				printf("\tSBC (0xF1) BCD not supported yet\n");
				exit (1);
			}
			tmpval = m6502.A;
			if (m6502.P & C_FLAG) tmpval++;
			tmpval += ABSOLUTE ^ 0xFF;
			m6502.P &= ~(N_FLAG | C_FLAG | V_FLAG | Z_FLAG);
			/* test for carry */
			if (tmpval & 0x100)
				m6502.P |= C_FLAG;
			tmpval &= 0xFF;
			/* test for Sign Bit overflow */
			if ((m6502.A ^ ABSOLUTE) &
				(m6502.A ^ tmpval) & 0x80)
				m6502.P |= V_FLAG;
			m6502.A = tmpval;
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 3;
			clockcnt += 4; /* plus one if it crosses page TODO */
			break;

		case 0xFD: /* SBC Absolute X (Subtract with Carry) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSBC $%02X%02X,X (%02X) ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1],
		ABSOLUTEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (m6502.P & D_FLAG) { /* is it decimal? */
				printf("\tSBC (0xF1) BCD not supported yet\n");
				exit (1);
			}
			tmpval = m6502.A;
			if (m6502.P & C_FLAG) tmpval++;
			tmpval += ABSOLUTEX ^ 0xFF;
			m6502.P &= ~(N_FLAG | C_FLAG | V_FLAG | Z_FLAG);
			/* test for carry */
			if (tmpval & 0x100)
				m6502.P |= C_FLAG;
			tmpval &= 0xFF;
			/* test for Sign Bit overflow */
			if ((m6502.A ^ ABSOLUTEX) &
				(m6502.A ^ tmpval) & 0x80)
				m6502.P |= V_FLAG;
			m6502.A = tmpval;
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0xF9: /* SBC Absolute Y (Subtract with Carry) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSBC $%02X%02X,Y (%02X) ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1],
		ABSOLUTEY);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (m6502.P & D_FLAG) { /* is it decimal? */
				printf("\tSBC (0xF1) BCD not supported yet\n");
				exit (1);
			}
			tmpval = m6502.A;
			if (m6502.P & C_FLAG) tmpval++;
			tmpval += ABSOLUTEY ^ 0xFF;
			m6502.P &= ~(N_FLAG | C_FLAG | V_FLAG | Z_FLAG);
			/* test for carry */
			if (tmpval & 0x100)
				m6502.P |= C_FLAG;
			tmpval &= 0xFF;
			/* test for Sign Bit overflow */
			if ((m6502.A ^ ABSOLUTEY) &
				(m6502.A ^ tmpval) & 0x80)
				m6502.P |= V_FLAG;
			m6502.A = tmpval;
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0xE1: /* SBC Indirect X (SuBtract from acc. with Carry) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSBC ($%02X,X) (%02X%02X) (%02X) ", mem_map[m6502.PC + 1],
		mem_map[mem_map[m6502.PC + 1] + m6502.X + 1],
		mem_map[mem_map[m6502.PC + 1] + m6502.X],
		INDIRECTX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (m6502.P & D_FLAG) { /* is it decimal? */
				printf("\tSBC (0xF1) BCD not supported yet\n");
				exit (1);
			}
			tmpval = m6502.A;
			if (m6502.P & C_FLAG) tmpval++;
			tmpval += INDIRECTX ^ 0xFF;
			m6502.P &= ~(N_FLAG | C_FLAG | V_FLAG | Z_FLAG);
			/* test for carry */
			if (tmpval & 0x100)
				m6502.P |= C_FLAG;
			tmpval &= 0xFF;
			/* test for Sign Bit overflow */
			if ((m6502.A ^ INDIRECTX) &
				(m6502.A ^ tmpval) & 0x80)
				m6502.P |= V_FLAG;
			m6502.A = tmpval;
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 6;
			break;

		case 0xF1: /* SBC Indirect Y (Subtract with Carry) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	tmpval = mem_map[mem_map[m6502.PC + 1]] + m6502.Y +
		(mem_map[(mem_map[m6502.PC + 1] + 1) & 0xFF] << 8);
	printf("\tSBC ($%02X),Y (%04X) (%02X) ", mem_map[m6502.PC + 1],
		tmpval, INDIRECTY);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (m6502.P & D_FLAG) { /* is it decimal? */
				printf("\tSBC (0xF1) BCD not supported yet\n");
				exit (1);
			}
			tmpval = m6502.A;
			if (m6502.P & C_FLAG) tmpval++;
			tmpval += INDIRECTY ^ 0xFF;
			m6502.P &= ~(N_FLAG | C_FLAG | V_FLAG | Z_FLAG);
			/* test for carry */
			if (tmpval & 0x100)
				m6502.P |= C_FLAG;
			tmpval &= 0xFF;
			/* test for Sign Bit overflow */
			if ((m6502.A ^ INDIRECTY) &
				(m6502.A ^ tmpval) & 0x80)
				m6502.P |= V_FLAG;
			m6502.A = tmpval;
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 5;
			break;

		case 0xF2: /* SBC (ZP) (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	tmpval = mem_map[mem_map[m6502.PC + 1]] +
		(mem_map[(mem_map[m6502.PC + 1] + 1) & 0xFF] << 8);
	printf("\tSBC ($%02X) (%04X) (%02X) ", mem_map[m6502.PC + 1],
		tmpval, INDIRECTZ);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			if (m6502.P & D_FLAG) { /* is it decimal? */
				printf("\tSBC (0xF2) BCD not supported yet\n");
				exit (1);
			}
			tmpval = m6502.A;
			if (m6502.P & C_FLAG) tmpval++;
			tmpval += INDIRECTZ ^ 0xFF;
			m6502.P &= ~(N_FLAG | C_FLAG | V_FLAG | Z_FLAG);
			/* test for carry */
			if (tmpval & 0x100)
				m6502.P |= C_FLAG;
			tmpval &= 0xFF;
			/* test for Sign Bit overflow */
			if ((m6502.A ^ INDIRECTZ) &
				(m6502.A ^ tmpval) & 0x80)
				m6502.P |= V_FLAG;
			m6502.A = tmpval;
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 5;
			break;

/***************************
 * End of SBC Instructions *
 ***************************/

/********************
 * ROL Instructions *
 ********************/

		case 0x2A: /* ROL Accumulator (rotate left) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tROL A ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			tmpval = (m6502.A << 1) + ((m6502.P & C_FLAG) ? 1 : 0);
			m6502.P &= ~(N_FLAG | C_FLAG | Z_FLAG);
			m6502.A = tmpval;
			if (tmpval & 0x100)
				m6502.P |= C_FLAG;
			m6502.P |= N_FLAG & tmpval; /* is it negitive */
			if (!(m6502.A & 0xFF))
				m6502.P |= Z_FLAG;
			m6502.PC += 1;
			clockcnt += 2;
			break;

		case 0x26: /* ROL ZeroPage (rotate left) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tROL $%02X (%02X) ", mem_map[m6502.PC + 1], ZEROPAGE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			tmpval = (ZEROPAGE << 1) + ((m6502.P & C_FLAG) ? 1 : 0);
			m6502.P &= ~(C_FLAG | N_FLAG | Z_FLAG);
			ZEROPAGE = tmpval;
			if (tmpval & 0x100)
				m6502.P |= C_FLAG;
			m6502.P |= N_FLAG & tmpval; /* is it negitive */
			if (!(tmpval & 0xFF))
				m6502.P |= Z_FLAG;
			m6502.PC += 2;
			clockcnt += 5;
			break;

		case 0x36: /* ROL ZeroPage X (ROtate Left one bit) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tROL $%02X,X (%02X) ", mem_map[m6502.PC + 1], ZEROPAGEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			tmpval = (ZEROPAGEX << 1) + ((m6502.P & C_FLAG) ? 1 : 0);
			m6502.P &= ~(C_FLAG | N_FLAG | Z_FLAG);
			ZEROPAGEX = tmpval;
			if (tmpval & 0x100)
				m6502.P |= C_FLAG;
			m6502.P |= N_FLAG & tmpval; /* is it negitive */
			if (!(tmpval & 0xFF))
				m6502.P |= Z_FLAG;
			m6502.PC += 2;
			clockcnt += 6;
			break;

		case 0x2E: /* ROL Absolute (ROtate Left one bit) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tROL $%02X%02X (%02X) ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1], ABSOLUTE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			tmpval = (ABSOLUTE << 1) + ((m6502.P & C_FLAG) ? 1 : 0);
			m6502.P &= ~(C_FLAG | N_FLAG | Z_FLAG);
			ABSOLUTE = tmpval;
			if (tmpval & 0x100)
				m6502.P |= C_FLAG;
			m6502.P |= N_FLAG & tmpval; /* is it negitive */
			if (!(tmpval & 0xFF))
				m6502.P |= Z_FLAG;
			m6502.PC += 3;
			clockcnt += 6;
			break;

		case 0x3E: /* ROL Absolute X (ROtate Left one bit) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tROL $%02X%02X,X (%02X) ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1], ABSOLUTEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			tmpval = (ABSOLUTEX << 1) + ((m6502.P & C_FLAG) ? 1 : 0);
			m6502.P &= ~(C_FLAG | N_FLAG | Z_FLAG);
			ABSOLUTEX = tmpval;
			if (tmpval & 0x100)
				m6502.P |= C_FLAG;
			m6502.P |= N_FLAG & tmpval; /* is it negitive */
			if (!(tmpval & 0xFF))
				m6502.P |= Z_FLAG;
			m6502.PC += 3;
			clockcnt += 7;
			break;

/***************************
 * End of ROL Instructions *
 ***************************/

/********************
 * ROR Instructions *
 ********************/

		case 0x6A: /* ROR Accumulator (rotate right) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tROR A ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			tmpval = (m6502.A >> 1) + ((m6502.P & C_FLAG) ? 0x80 : 0);
			m6502.P &= ~(C_FLAG | N_FLAG | Z_FLAG);
			m6502.P |= m6502.A & C_FLAG;
			m6502.A = tmpval;
			m6502.P |= m6502.A & N_FLAG; /* is it negitive */
			if (!m6502.A)
				m6502.P |= Z_FLAG;
			m6502.PC += 1;
			clockcnt += 2;
			break;

		case 0x66: /* ROR ZeroPage (rotate right) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tROR $%02X (%02X) ", mem_map[m6502.PC + 1], ZEROPAGE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			tmpval = (ZEROPAGE >> 1) + ((m6502.P & C_FLAG) ? 0x80 : 0);
			m6502.P &= ~(C_FLAG | N_FLAG | Z_FLAG);
			m6502.P |= ZEROPAGE & C_FLAG;
			ZEROPAGE = tmpval;
			m6502.P |= ZEROPAGE & N_FLAG; /* is it negitive */
			if (!ZEROPAGE)
				m6502.P |= Z_FLAG;
			m6502.PC += 2;
			clockcnt += 5;
			break;

		case 0x76: /* ROR ZeroPage X (ROtate Right one bit) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tROR $%02X,X (%02X) ", mem_map[m6502.PC + 1], ZEROPAGEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			tmpval = (ZEROPAGEX >> 1) + ((m6502.P & C_FLAG) ? 0x80 : 0);
			m6502.P &= ~(C_FLAG | N_FLAG | Z_FLAG);
			m6502.P |= ZEROPAGEX & C_FLAG;
			ZEROPAGEX = tmpval;
			m6502.P |= ZEROPAGEX & N_FLAG; /* is it negitive */
			if (!ZEROPAGEX)
				m6502.P |= Z_FLAG;
			m6502.PC += 2;
			clockcnt += 6;
			break;

		case 0x6E: /* ROR Absolute (ROtate Right one bit) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tROR $%02X%02X (%02X) ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1], ABSOLUTE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			tmpval = (ABSOLUTE >> 1) + ((m6502.P & C_FLAG) ? 0x80 : 0);
			m6502.P &= ~(C_FLAG | N_FLAG | Z_FLAG);
			m6502.P |= ABSOLUTE & C_FLAG;
			ABSOLUTE = tmpval;
			m6502.P |= ABSOLUTE & N_FLAG; /* is it negitive */
			if (!ABSOLUTE)
				m6502.P |= Z_FLAG;
			m6502.PC += 3;
			clockcnt += 6;
			break;

		case 0x7E: /* ROR Absolute X (ROtate Right one bit) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tROR $%02X%02X,X (%02X) ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1], ABSOLUTEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			tmpval = (ABSOLUTEX >> 1) + ((m6502.P & C_FLAG) ? 0x80 : 0);
			m6502.P &= ~(C_FLAG | N_FLAG | Z_FLAG);
			m6502.P |= ABSOLUTEX & C_FLAG;
			ABSOLUTEX = tmpval;
			m6502.P |= ABSOLUTEX & N_FLAG; /* is it negitive */
			if (!ABSOLUTEX)
				m6502.P |= Z_FLAG;
			m6502.PC += 3;
			clockcnt += 7;

/***************************
 * End of ROR Instructions *
 ***************************/

/********************
 * LSR Instructions *
 ********************/

		case 0x4A: /* LSR Accumulator, logical shift right */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tLSR A ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			/* high bit is always zero, so N_FLAG is always cleared */
			m6502.P &= ~(N_FLAG | C_FLAG | Z_FLAG);
			m6502.P |= m6502.A & C_FLAG; /* the bottom bit goes to carry */
			m6502.A = m6502.A >> 1;
			if (!m6502.A) /* is A zero? */
				m6502.P |= Z_FLAG;
			m6502.PC += 1;
			clockcnt += 2;
			break;

		case 0x46: /* LSR ZeroPage, logical shift right */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tLSR $%02X (%02X) ", mem_map[m6502.PC + 1], ZEROPAGE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(N_FLAG | C_FLAG | Z_FLAG);
			m6502.P |= ZEROPAGE & C_FLAG ;
			ZEROPAGE = ZEROPAGE >> 1;
			if (!ZEROPAGE)
				m6502.P |= Z_FLAG;
			m6502.PC += 2;
			clockcnt += 5;
			break;

		case 0x56: /* LSR ZeroPage X, (Logical Shift Right on bit) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tLSR $%02X,X (%02X) ", mem_map[m6502.PC + 1], ZEROPAGEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(N_FLAG | C_FLAG | Z_FLAG);
			m6502.P |= ZEROPAGEX & C_FLAG;
			ZEROPAGEX = ZEROPAGEX >> 1;
			if (!ZEROPAGEX)
				m6502.P |= Z_FLAG;
			m6502.PC += 2;
			clockcnt += 6;
			break;

		case 0x4E: /* LSR Absolute, (Logical Shift Right on bit) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tLSR $%02X%02X (%02X) ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1],
		ABSOLUTE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(N_FLAG | C_FLAG | Z_FLAG);
			m6502.P |= ABSOLUTE & C_FLAG;
			ABSOLUTE = ABSOLUTE >> 1;
			if (!ABSOLUTE)
				m6502.P |= Z_FLAG;
			m6502.PC += 3;
			clockcnt += 6;
			break;

		case 0x5E: /* LSR Absolute X, (Logical Shift Right on bit) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tLSR $%02X%02X,X (%02X) ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1],
		ABSOLUTEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(N_FLAG | C_FLAG | Z_FLAG);
			m6502.P |= ABSOLUTEX & C_FLAG;
			ABSOLUTEX = ABSOLUTEX >> 1;
			if (!ABSOLUTEX)
				m6502.P |= Z_FLAG;
			m6502.PC += 3;
			clockcnt += 7;
			break;

/***************************
 * End of LSR Instructions *
 ***************************/

/********************
 * AND Instructions *
 ********************/

		case 0x29: /* AND Immediate (AND memory with accumulator) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tAND #$%02X ", IMMEDIATE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A &= IMMEDIATE;
			m6502.P &= ~(N_FLAG | Z_FLAG);
			if (!m6502.A)
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A;
			m6502.PC += 2;
			clockcnt += 2;
			break;

		case 0x25: /* AND ZeroPage (AND memory with acc.) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tAND $%02X (%02X)", mem_map[m6502.PC + 1], ZEROPAGE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A &= ZEROPAGE;
			m6502.P &= ~(N_FLAG | Z_FLAG);
			if (!m6502.A)
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A;
			m6502.PC += 2;
			clockcnt += 2;
			break;

		case 0x35: /* AND ZeroPage X, (AND memory with acc.) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tAND $%02X,X (%02X)", mem_map[m6502.PC + 1], ZEROPAGEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A &= ZEROPAGEX;
			m6502.P &= ~(N_FLAG | Z_FLAG);
			if (!m6502.A)
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A;
			m6502.PC += 2;
			clockcnt += 3;
			break;

		case 0x2D: /* AND Absolute (AND memory with acc.) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tAND $%02X%02X (%02X)", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1],
		ABSOLUTE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A &= ABSOLUTE;
			m6502.P &= ~(N_FLAG | Z_FLAG);
			if (!m6502.A)
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A;
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0x3D: /* AND Absolute X (AND memory with acc.) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tAND $%02X%02X,X (%02X)", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1],
		ABSOLUTEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A &= ABSOLUTEX;
			m6502.P &= ~(N_FLAG | Z_FLAG);
			if (!m6502.A)
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A;
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0x39: /* AND Absolute Y (AND memory with acc.) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tAND $%02X%02X,Y (%02X)", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1],
		ABSOLUTEY);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A &= ABSOLUTEY;
			m6502.P &= ~(N_FLAG | Z_FLAG);
			if (!m6502.A)
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A;
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0x21: /* AND Indirect X (AND memory with accumulator) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tAND ($%02X,X) (%02X%02X) (%02X) ", mem_map[m6502.PC + 1],
		mem_map[mem_map[m6502.PC + 1] + m6502.X + 1],
		mem_map[mem_map[m6502.PC + 1] + m6502.X],
		INDIRECTX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A &= INDIRECTX;
			m6502.P &= ~(N_FLAG | Z_FLAG);
			if (!m6502.A)
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A;
			m6502.PC += 2;
			clockcnt += 6;
			break;

		case 0x31: /* AND Indirect Y (AND memory with accumulator) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	tmpval = mem_map[mem_map[m6502.PC + 1]] + m6502.Y +
		(mem_map[(mem_map[m6502.PC + 1] + 1) & 0xFF] << 8);
	printf("\tAND ($%02X),Y (%04X) (%02X) ", mem_map[m6502.PC + 1],
		tmpval, INDIRECTY);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A &= INDIRECTY;
			m6502.P &= ~(N_FLAG | Z_FLAG);
			if (!m6502.A)
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A;
			m6502.PC += 2;
			clockcnt += 5;
			break;

		case 0x32: /* AND (ZP) (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tAND ($%02X) (%02X%02X) (%02X) ", mem_map[m6502.PC + 1],
		0xFF & mem_map[mem_map[m6502.PC + 1] + 1],
		mem_map[mem_map[m6502.PC + 1]],
		INDIRECTZ);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A &= INDIRECTZ;
			m6502.P &= ~(N_FLAG | Z_FLAG);
			if (!m6502.A)
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A;
			m6502.PC += 2;
			clockcnt += 6;
			break;

/***************************
 * End of AND Instructions *
 ***************************/

/********************
 * ORA Instructions *
 ********************/

		case 0x09: /* ORA Immediate (OR memory with Acc.) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tORA #$%02X ", IMMEDIATE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(N_FLAG | Z_FLAG);
			m6502.A |= IMMEDIATE;
			if (!m6502.A)
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A;
			m6502.PC += 2;
			clockcnt += 2;
			break;

		case 0x05: /* ORA ZeroPage */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tORA $%02X (%02X) ", mem_map[m6502.PC + 1], ZEROPAGE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(N_FLAG | Z_FLAG);
			m6502.A |= ZEROPAGE;
			if (!m6502.A)
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A;
			m6502.PC += 2;
			clockcnt += 2;
			break;

		case 0x15: /* ORA ZeroPage X, (OR memory with Acc.) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tORA $%02X,X (%02X) ", mem_map[m6502.PC + 1], ZEROPAGEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(N_FLAG | Z_FLAG);
			m6502.A |= ZEROPAGEX;
			if (!m6502.A)
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A;
			m6502.PC += 2;
			clockcnt += 2;
			break;

		case 0x0D: /* ORA Absolute */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tORA $%02X%02X ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(N_FLAG | Z_FLAG);
			m6502.A |= ABSOLUTE;
			if (!m6502.A)
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A;
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0x1D: /* ORA Absolute X */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tORA $%02X%02X,X ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(N_FLAG | Z_FLAG);
			m6502.A |= ABSOLUTEX;
			if (!m6502.A)
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A;
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0x19: /* ORA Absolute Y */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tORA $%02X%02X,Y ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(N_FLAG | Z_FLAG);
			m6502.A |= ABSOLUTEY;
			if (!m6502.A)
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A;
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0x01: /* ORA Indirect X (OR memory with Acc.) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tORA ($%02X,X) (%02X%02X) (%02X) ", mem_map[m6502.PC + 1],
		mem_map[mem_map[m6502.PC + 1] + m6502.X + 1],
		mem_map[mem_map[m6502.PC + 1] + m6502.X],
		INDIRECTX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(N_FLAG | Z_FLAG);
			m6502.A |= INDIRECTX;
			if (!m6502.A)
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A;
			m6502.PC += 2;
			clockcnt += 6;
			break;

		case 0x11: /* ORA Indirect Y (OR memory with Acc.) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	tmpval = mem_map[mem_map[m6502.PC + 1]] + m6502.Y +
		(mem_map[(mem_map[m6502.PC + 1] + 1) & 0xFF] << 8);
	printf("\tORA ($%02X),Y (%04X) (%02X) ", mem_map[m6502.PC + 1],
		tmpval, INDIRECTY);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(N_FLAG | Z_FLAG);
			m6502.A |= INDIRECTY;
			if (!m6502.A)
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A;
			m6502.PC += 2;
			clockcnt += 5;
			break;

		case 0x12: /* ORA (ZP) (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	tmpval = ((0xFF & mem_map[mem_map[m6502.PC + 1] + 1]) << 8) +
		mem_map[mem_map[m6502.PC + 1]];
	printf("\tORA ($%02X) (%04X) (%02X) ", mem_map[m6502.PC + 1],
		tmpval, INDIRECTZ);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(N_FLAG | Z_FLAG);
			m6502.A |= INDIRECTZ;
			if (!m6502.A)
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A;
			m6502.PC += 2;
			clockcnt += 5;
			break;

/***************************
 * End of ORA Instructions *
 ***************************/

/********************
 * INC Instructions *
 ********************/

		case 0xE6: /* INC ZeroPage */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tINC $%02X (%02X) ", mem_map[m6502.PC + 1], ZEROPAGE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			ZEROPAGE++;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!ZEROPAGE) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & ZEROPAGE; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 5;
			break;

		case 0xF6: /* INC ZeroPage X, (INCrement mem by one) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tINC $%02X,X (%02X) ", mem_map[m6502.PC + 1], ZEROPAGEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			ZEROPAGEX++;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!ZEROPAGEX) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & ZEROPAGEX; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 6;
			break;

		case 0xEE: /* INC Absolute */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tINC $%02X%02X (%02X) ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1],
		ABSOLUTE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			ABSOLUTE++;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!ABSOLUTE) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & ABSOLUTE; /* is it negitive? */
			m6502.PC += 3;
			clockcnt += 6;
			break;

		case 0xFE: /* INC Absolute X */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tINC $%02X%02X,X (%02X) ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1],
		ABSOLUTEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			ABSOLUTEX++;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!ABSOLUTEX) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & ABSOLUTEX; /* is it negitive? */
			m6502.PC += 3;
			clockcnt += 7;
			break;

		case 0x1A: /* INC A (INA) (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tINC A ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A++;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 1;
			clockcnt += 2;
			break;

/***************************
 * End of INC Instructions *
 ***************************/

/********************
 * DEC Instructions *
 ********************/

		case 0xC6: /* DEC ZeroPage */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tDEC $%02X (%02X) ", mem_map[m6502.PC + 1], ZEROPAGE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			ZEROPAGE--;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!ZEROPAGE) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & ZEROPAGE; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 5;
			break;

		case 0xD6: /* DEC ZeroPage X, (DECrement memory by one) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tDEC $%02X,X (%02X) ", mem_map[m6502.PC + 1], ZEROPAGEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			ZEROPAGEX--;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!ZEROPAGEX) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & ZEROPAGEX; /* is it negitive? */
			m6502.PC += 2;
			clockcnt += 6;
			break;

		case 0xCE: /* DEC Absolute */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tDEC $%02X%02X (%02X) ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1], ABSOLUTE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			ABSOLUTE--;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!ABSOLUTE) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & ABSOLUTE; /* is it negitive? */
			m6502.PC += 3;
			clockcnt += 6;
			break;

		case 0xDE: /* DEC Absolute X */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tDEC $%02X%02X,X (%02X) ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1], ABSOLUTEX);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			ABSOLUTEX--;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!ABSOLUTEX) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & ABSOLUTEX; /* is it negitive? */
			m6502.PC += 3;
			clockcnt += 7;
			break;

		case 0x3A: /* DEC A (DEA) (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tDEC A ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.A--;
			m6502.P &= ~(Z_FLAG | N_FLAG);
			if (!m6502.A) /* is it zero? */
				m6502.P |= Z_FLAG;
			m6502.P |= N_FLAG & m6502.A; /* is it negitive? */
			m6502.PC += 1;
			clockcnt += 2;
			break;


/***************************
 * End of DEC Instructions *
 ***************************/

		case 0xEA: /* NOP (No OPeration, do nothing) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tNOP ");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.PC += 1;
			clockcnt += 2;
			break;

/*************************
 * 65C02 Invalid OpCodes *
 *************************/

		case 0x02: /* Invalid NOP */
		case 0x22: /* Invalid NOP */
		case 0x42: /* Invalid NOP */
		case 0x62: /* Invalid NOP */
		case 0x82: /* Invalid NOP */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tInvalid NOP %02X ", mem_map[m6502.PC]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.PC += 2;
			clockcnt += 2;
			break;

		case 0xC2: /* Invalid NOP */
		case 0xE2: /* Invalid NOP */
		case 0x03: /* Invalid NOP */
		case 0x13: /* Invalid NOP */
		case 0x23: /* Invalid NOP */
		case 0x33: /* Invalid NOP */
		case 0x43: /* Invalid NOP */
		case 0x53: /* Invalid NOP */
		case 0x63: /* Invalid NOP */
		case 0x73: /* Invalid NOP */
		case 0x83: /* Invalid NOP */
		case 0x93: /* Invalid NOP */
		case 0xA3: /* Invalid NOP */
		case 0xB3: /* Invalid NOP */
		case 0xC3: /* Invalid NOP */
		case 0xD3: /* Invalid NOP */
		case 0xE3: /* Invalid NOP */
		case 0xF3: /* Invalid NOP */
		case 0x0B: /* Invalid NOP */
		case 0x1B: /* Invalid NOP */
		case 0x2B: /* Invalid NOP */
		case 0x3B: /* Invalid NOP */
		case 0x4B: /* Invalid NOP */
		case 0x5B: /* Invalid NOP */
		case 0x6B: /* Invalid NOP */
		case 0x7B: /* Invalid NOP */
		case 0x8B: /* Invalid NOP */
		case 0x9B: /* Invalid NOP */
		case 0xAB: /* Invalid NOP */
		case 0xBB: /* Invalid NOP */
		case 0xEB: /* Invalid NOP */
		case 0xFB: /* Invalid NOP */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tInvalid NOP %02X ", mem_map[m6502.PC]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.PC += 1;
			clockcnt += 1;
			break;

		case 0x44: /* Invalid NOP */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tInvalid NOP %02X ", mem_map[m6502.PC]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.PC += 2;
			clockcnt += 3;
			break;

		case 0x54: /* Invalid NOP */
		case 0xD4: /* Invalid NOP */
		case 0xF4: /* Invalid NOP */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tInvalid NOP %02X ", mem_map[m6502.PC]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.PC += 2;
			clockcnt += 4;
			break;

		case 0x5C: /* Invalid NOP */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tInvalid NOP %02X ", mem_map[m6502.PC]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.PC += 3;
			clockcnt += 8;
			break;

		case 0xDC: /* Invalid NOP */
		case 0xFC: /* Invalid NOP */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tInvalid NOP %02X ", mem_map[m6502.PC]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.PC += 3;
			clockcnt += 4;
			break;

/********************************
 * End of 65C02 Invalid OpCodes *
 ********************************/

/**********************
 * 65C02 Only OpCodes *
 **********************/

/* BBR, BBS, RMB, SMB, STP, WAI, STZ, PHX, PLX, PHY, PLY, TSB, TRB, BRA */

		case 0x74: /* STZ ZeroPage X (STore Zero in memory) (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSTZ $%02X,X ", mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			ZEROPAGEX = 0;
			m6502.PC += 2;
			clockcnt += 3;
			break;

		case 0x64: /* STZ ZeroPage (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSTZ $%02X ", mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			ZEROPAGE = 0;
			m6502.PC += 2;
			clockcnt += 3;
			break;

		case 0x9C: /* STZ Absolute (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSTZ $%02X%02X ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			ABSOLUTE = 0;
			m6502.PC += 3;
			clockcnt += 4;
			break;

		case 0x9E: /* STZ A,X (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSTZ $%02X%02X,X ", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			ABSOLUTEX = 0;
			m6502.PC += 3;
			clockcnt += 5;
			break;

		case 0x04: /* TSB ZeroPage (Test and Set Bits) (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tTSB $%02X", mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(Z_FLAG);
			if (ZEROPAGE & m6502.A)
				m6502.P |= Z_FLAG;
			ZEROPAGE |= m6502.A;
			m6502.PC += 2;
			clockcnt += 5;
			break;
		
		case 0x0C: /* TSB Absolute (Test and Set Bits) (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tTSB $%02X%02X", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(Z_FLAG);
			if (ABSOLUTE & m6502.A)
				m6502.P |= Z_FLAG;
			ABSOLUTE |= m6502.A;
			m6502.PC += 3;
			clockcnt += 6;
			break;

		case 0x14: /* TRB ZeroPage (Test and Reset Bits) (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tTRB $%02X", mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(Z_FLAG);
			if (ZEROPAGE & m6502.A)
				m6502.P |= Z_FLAG;
			ZEROPAGE &= ~m6502.A;
			m6502.PC += 2;
			clockcnt += 5;
			break;
		
		case 0x1C: /* TRB Absolute (Test and Reset Bits) (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tTRB $%02X%02X", mem_map[m6502.PC + 2], mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			m6502.P &= ~(Z_FLAG);
			if (ABSOLUTE & m6502.A)
				m6502.P |= Z_FLAG;
			ABSOLUTE &= ~m6502.A;
			m6502.PC += 3;
			clockcnt += 6;
			break;

		case 0x07: /* RMB ZeroPage (65C02 Only) */
		case 0x17: /* RMB ZeroPage (65C02 Only) */
		case 0x27: /* RMB ZeroPage (65C02 Only) */
		case 0x37: /* RMB ZeroPage (65C02 Only) */
		case 0x47: /* RMB ZeroPage (65C02 Only) */
		case 0x57: /* RMB ZeroPage (65C02 Only) */
		case 0x67: /* RMB ZeroPage (65C02 Only) */
		case 0x77: /* RMB ZeroPage (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tRMB%01X $%02X ", mem_map[m6502.PC] >> 4, mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			tmpval = 1 << (mem_map[m6502.PC] >> 4);
			mem_map[mem_map[m6502.PC + 1]] &= ~tmpval;
			m6502.PC += 2;
			clockcnt += 2;
			break;

		case 0x87: /* SMB ZeroPage (65C02 Only) */
		case 0x97: /* SMB ZeroPage (65C02 Only) */
		case 0xA7: /* SMB ZeroPage (65C02 Only) */
		case 0xB7: /* SMB ZeroPage (65C02 Only) */
		case 0xC7: /* SMB ZeroPage (65C02 Only) */
		case 0xD7: /* SMB ZeroPage (65C02 Only) */
		case 0xE7: /* SMB ZeroPage (65C02 Only) */
		case 0xF7: /* SMB ZeroPage (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tSMB%01X $%02X ", (mem_map[m6502.PC] >> 4) - 7, mem_map[m6502.PC + 1]);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			tmpval = 1 << ((mem_map[m6502.PC] >> 4) - 8);
			mem_map[mem_map[m6502.PC + 1]] |= tmpval;
			m6502.PC += 2;
			clockcnt += 2;
			break;

		case 0x8F: /* BBS0 (65C02 Only) */
		case 0x9F: /* BBS1 (65C02 Only) */
		case 0xAF: /* BBS2 (65C02 Only) */
		case 0xBF: /* BBS3 (65C02 Only) */
		case 0xCF: /* BBS4 (65C02 Only) */
		case 0xDF: /* BBS5 (65C02 Only) */
		case 0xEF: /* BBS6 (65C02 Only) */
		case 0xFF: /* BBS7 (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tBBS%01X $%02X,$%02X (%02X)", (mem_map[m6502.PC] >> 4) - 7, mem_map[m6502.PC + 1],
	mem_map[m6502.PC + 2], ZEROPAGE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			tmpval = 1 << ((mem_map[m6502.PC] >> 4) - 8);
			if (tmpval & ZEROPAGE) {
				m6502.PC += 3 + (char) mem_map[m6502.PC + 2];
			} else
				 m6502.PC += 3;
			clockcnt += 2;
			break;

		case 0x0F: /* BBR0 (65C02 Only) */
		case 0x1F: /* BBR1 (65C02 Only) */
		case 0x2F: /* BBR2 (65C02 Only) */
		case 0x3F: /* BBR3 (65C02 Only) */
		case 0x4F: /* BBR4 (65C02 Only) */
		case 0x5F: /* BBR5 (65C02 Only) */
		case 0x6F: /* BBR6 (65C02 Only) */
		case 0x7F: /* BBR7 (65C02 Only) */
			if (trace_cpu >= 2) {
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\tBBR%01X $%02X,$%02X (%02X)", (mem_map[m6502.PC] >> 4) - 7, mem_map[m6502.PC + 1],
	mem_map[m6502.PC + 2], ZEROPAGE);
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			}
			tmpval = 1 << ((mem_map[m6502.PC] >> 4) - 8);
			if (tmpval & ZEROPAGE) {
				 m6502.PC += 3;
			} else
				m6502.PC += 3 + (char) mem_map[m6502.PC + 2];
			clockcnt += 2;
			break;

/**********************************
 * End of 65C02 Only Instructions *
 **********************************/

#ifdef NOTDEF
	/* JGG uncoded opcodes here */
		case 0x5A: /* TODO: PHY S (65C02 Only) */
		case 0x7A: /* TODO: PLY S (65C02 Only) */
		case 0x80: /* TODO: BRA R (65C02 Only) */
		case 0xCB: /* TODO: WAI I (65C02 Only) */
		case 0xDA: /* TODO: PHX S (65C02 Only) */
		case 0xDB: /* TODO: STP I (65C02 Only) */
		case 0xFA: /* TODO: PLX S (65C02 Only) */
#endif


		default:
			if (use_ansi_color) printf("\033[01;31m"); /* bold red */
			printf("\n========================================\n");
			printf("I don't know opcode %02x yet!\n", mem_map[m6502.PC]);
			printf("========================================\n");
			if (use_ansi_color) printf("\033[22;37m"); /* normal white */
			exit(1);
	}

	if (dostep) {
		dostep = 0;
		dopause = 1;
	}
}
void
do_output(void)
{
	static char keylist[] =
	"@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_ !\"#$\%&'()*+,-./0123456789:;<=>?";

	/* this is set to A7 by the WOZMON on startup and reset. */
	if (mem_map[0xD011] == 0xA7) /* clear the keyboard strobe */
		mem_map[0xD011] = 0;
#ifdef NOTDEF
	/* 0xD010 = keyboard register */
	/*if (mem_map[0xD011] == 0xA7 && mem_map[0xD010] == 0x80) {*/ /* keyboard control register */
	/* remember, high bit of D010 always set */
		if (use_ansi_color) fprintf(stderr, "\033[01;34m"); /* bold blue */
		fprintf(stderr, "\n<keyboard initialized, irq ready>\n");
		if (use_ansi_color) fprintf(stderr, "\033[22;37m"); /* normal white */
		mem_map[0xD011] = 0x27; /* based on mess, probably wrong */
	}

/*
 * Note: high bit of D011 means key waiting in buffer
 * woz basic uses bit and BMI to stop running if key pressed
 * even if there isn't one. Need to check on real unit what
 * possible values of D011 exist when key not pressed.
 */

#endif

/* on xmess.x11 apple1, D012: XX
 * 21 = !
 * 31 = 1
 * 41 = A
 * 51 = Q
 * 61 = A
 * 71 = Q
 * A1 = !
 * B1 = 1
 * C1 = A
 * D1 = Q
 * E1 = A
 * F1 = Q
 * 01 11 81 91 do nothing, need to try this on real machines
 */

	mem_map[0xD012] &= 0x7F; /* turn off high bit */

	if (mem_map[0xD012] == 0x0D) { /* carrage return */
#ifdef DEBUG
		printf("\n<output CR>\n");
#else
		fprintf(stderr, "\n");
#endif
	} else if (mem_map[0xD012] & 0x40) { /* it's a letter */
		if (use_ansi_color) printf("\033[01;32m"); /* bold green */
#ifdef DEBUG
		printf("\n<output %02X %c>\n", mem_map[0xD012], keylist[mem_map[0xD012] & 0x1F]);
#else
		fprintf(stderr, "%c", keylist[mem_map[0xD012] & 0x1F]);
		if (trace_cpu) fprintf(stderr, "\n");
#endif
		if (use_ansi_color) printf("\033[22;37m"); /* normal white */
	} else if (mem_map[0xD012] & 0x20) { /* it's a number */
		if (use_ansi_color) printf("\033[01;32m"); /* bold green */
#ifdef DEBUG
			printf("\n<output %02X %c>\n", mem_map[0xD012], keylist[(mem_map[0xD012] + 0x1F) + 0x20]);
#else
			fprintf(stderr, "%c", keylist[(mem_map[0xD012] & 0x1F) + 0x20]);
			if (trace_cpu) fprintf(stderr, "\n");
#endif
		if (use_ansi_color) printf("\033[22;37m"); /* normal white */
	}
#ifdef DEBUG
	else {
		if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
		fprintf(stderr, "\n<output unknown char %02x>\n", mem_map[0xD012]);
		if (use_ansi_color) printf("\033[22;37m"); /* normal white */
	}
#endif

#ifdef DEBUG
	if (mem_map[0xD013]) { /* video control register */
		if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
		printf("<video control register (%02x)>\n", mem_map[0xD013]);
		if (use_ansi_color) printf("\033[22;37m"); /* normal white */
		mem_map[0xD013] = 0x00;
	}
#endif

}


void read_raw(unsigned char *dest, char *filename, int length)
{
	FILE *myfile;
	int filesize;
#ifdef DEBUG
	fprintf(stderr, "Reading file %s into memory %p of length %i\n", filename, dest, length);
#endif
	if ((myfile=fopen(filename,"rb")) == NULL) {
		fprintf(stderr, "\nCan't open file %s\n", filename);
		exit(-1);
	}

	filesize = fread(dest,1,length,myfile);
	if (filesize != length) {
		fprintf(stderr, "\nThere wasn't enough data in file %s\n", filename);
		exit(-1);
	}

	fclose(myfile);
#ifdef DEBUG
	fprintf(stderr, "Done reading file %s\n", filename);
#endif
}

void do_F1(void) /* Help */
{
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\n========================================\n");
	printf("Dinaao Is Not An Apple One.\n");
	printf("F1 = Help\n");
	printf("F2 = Toggle instruction trace: none, status, full\n");
	printf("F3 = Toggle throttle\n");
	/* printf("F4 = Toggle between 6502 and 65C02 cpu\n"); */
	printf("F8 = Toggle Pause/Unpause\n");
	printf("F5 = Step one instruction (also sets pause mode) \n");
	printf("F9 = Exit\n");
	printf("F7 = Toggle ansi color\n");
	/* printf("F10 = NMI, start at address found at FFFA-FFFB (lsb first) \n"); */
	/* printf("F11 = IRQ (can be blocked by I) ... FFFE-FFFF ...\n"); */
	printf("F12 = RESET, ... FFFC-FFFD ... (restart monitor)\n");
	printf("========================================\n");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
}

void do_F2(void) /* Turn trace on */
{
	trace_cpu++;
	if (trace_cpu >= 3)
		trace_cpu = 0;
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\n========================================\n");
	fprintf(stderr, "Trace is now %i\n", trace_cpu);
	printf("========================================\n");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
}

void do_F3(void) /* Throttle */
{
	throttle = 1 - throttle;
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("========================================\n");
	fprintf(stderr, "Throttle is now %s\n", throttle ? "on" : "off");
	printf("========================================\n");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
}

void do_F4(void)
{
}

void do_F5(void) /* Step one instruction */
{
	dostep = 1;
	dopause = 0;
}

void do_F6(void)
{
}

void do_F7(void)
{
	use_ansi_color = 1 - use_ansi_color;
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("\n========================================\n");
	fprintf(stderr, "ANSI Color is now %i\n", use_ansi_color);
	printf("========================================\n");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
}

void do_F8(void) /* Pause */
{
	dopause = 1 - dopause;
	if (use_ansi_color) printf("\033[01;34m"); /* bold blue */
	printf("========================================\n");
	fprintf(stderr, "Pause is now %s\n", dopause ? "on" : "off");
	printf("========================================\n");
	if (use_ansi_color) printf("\033[22;37m"); /* normal white */
}

void do_F9(void) /* Exit */
{
	printf("\n");
	exit(0);
}

void do_F10(void) /* NMI */
{
	pending_irq |= DO_NMI;
}

void do_F11(void) /* IRQ */
{
	pending_irq |= DO_IRQ;
}

void do_F12(void) /* Reset */
{
	pending_irq |= DO_RESET;
	/* if cpu thread is waiting for a key, we need to unstick emu */
	/* so put something in the que */
	PutKey(0x00);
}

void do_INS(void)
{
}

void do_DEL(void)
{
}
